/*
 * Host AP (software wireless LAN access point) driver for Intersil Prism2
 * Copyright (c) 2001, SSH Communications Security Corp
 * Jouni Malinen <jkm@ssh.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. See README and COPYING for
 * more details.
 *
 * FIX:
 * - do we really need TX/TXEXC events? possibly not in the final version since
 *   alloc events can be used to release txfids; TX packet/byte counts for
 *   really transmitted frames might be one reason, or a need to react somehow
 *   to not-ACK'ed frames
 * - shutting down pcmcia-cs while wlan# interface is up does not work
 *   (it does not remove the driver and eventually the kernel will panic when
 *   card is removed); when interface is first configured down, pcmcia-cs can
 *   be shut down; update: this seems to work at least with kernel-based
 *   pcmcia and Linux 2.4.13..
 *
 * Buffer Access Path (BAP) usage:
 *   BAP0 is used for sending data from the host computer to the card and for
 *   RID read/writes; it is protected with local->baplock (spin_lock_bh) and it
 *   must not be used from hardware interrupt context.
 *   BAP1 is used for receiving data from the card to the host computer; it is
 *   used only in hardware interrupt handler and kernel is assumed to not call
 *   the same handler simultaneously for the same interrupt even on SMP
 *   systems (this removes need for spin_lock protecting BAP1 access.
 */


#include <linux/config.h>
#include <linux/version.h>

/* define PRISM2_MONITOR to add support for raw WLAN frame sniffing */
#define PRISM2_MONITOR

/* define PRISM2_NULLFUNC_ACK to 'fix' problems with Prism2 f/w 0.8.0 (at least
 * from Compaq); this f/w does not seem to ACK data::nullfunc frames, but
 * 802.11 requires that these frames are ACKed and at least Lucent WaveLAN
 * needs this control::ack when setting up PS mode; this 'fix' tries to send
 * the control::ack from kernel driver, and usually manages to match a
 * data::nullfunc frame eventually..
 * f/w upgrade would be a proper fix for this.. */
#define PRISM2_NULLFUNC_ACK


#ifndef __IN_PCMCIA_PACKAGE__

/* Kernel tree PCMCIA compilation */
#include <asm/uaccess.h>
#include <linux/wireless.h>

#define add_rx_bytes(stats, n) do { (stats)->rx_bytes += n; } while (0)
#define add_tx_bytes(stats, n) do { (stats)->tx_bytes += n; } while (0)
#define init_dev_name(dev, node) do { } while (0)
#define copy_dev_name(node, dev) strcpy((node).dev_name, (dev)->name)

#else /* __IN_PCMCIA_PACKAGE__ */

/* External pcmcia-cs compilation */
#include <pcmcia/config.h>
#include <pcmcia/k_compat.h>

#endif /* __IN_PCMCIA_PACKAGE__ */


#include <asm/io.h>
#include <asm/delay.h>

#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>

#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/cisreg.h>
#include <pcmcia/ds.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/wireless.h>
#include <linux/proc_fs.h>
#include <linux/if_arp.h>

#ifdef PRISM2_MONITOR
#ifdef __IN_PCMCIA_PACKAGE__
/* net/sock.h (at least in 2.2.17) does not like min()/max() macros from
 * pcmcia-cs's kernel compat header */
#ifdef min
#undef min
#endif
#ifdef max
#undef max
#endif
#endif /* __IN_PCMCIA_PACKAGE__ */
#include <net/sock.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
#endif /* PRISM2_MONITOR */
#include <linux/delay.h>

/* 2.2 compatibility */
#ifndef spin_lock_bh
#define spin_lock_bh(lock) spin_lock_irq(lock)
#define spin_unlock_bh(lock) spin_unlock_irq(lock)
#endif

#ifndef rtnl_shlock_nowait 
static struct semaphore rtnl_sem = MUTEX;
#define rtnl_shlock_nowait()    down_trylock(&rtnl_sem)
#endif
/* end 2.2 compatibility */


#include "prism2_wlan.h"
#include "prism2_ap.h"


/* #define final_version */


static char *version =
"prism2.c 0.0.0 2001-12-10 (SSH Communications Security Corp, Jouni Malinen)";
static dev_info_t dev_info = "prism2";
static dev_link_t *dev_list = NULL;
static struct proc_dir_entry *prism2_proc = NULL;

MODULE_AUTHOR("SSH Communications Security Corp, Jouni Malinen");
MODULE_DESCRIPTION("Support for Intersil Prism2-based 802.11 wireless LAN "
		   "cards.");
MODULE_SUPPORTED_DEVICE("Intersil Prism2-based WLAN cards");
#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif


static int mtu = 1500;
MODULE_PARM(mtu, "i");
MODULE_PARM_DESC(mtu, "Maximum transfer unit");


static unsigned int irq_mask = 0xdeb8;
MODULE_PARM(irq_mask, "i");

static int irq_list[4] = { -1 };
MODULE_PARM(irq_list, "1-4i");



static int channel = 3;
MODULE_PARM(channel, "i");

static char essid[MAX_SSID_LEN + 1] = "test";
MODULE_PARM(essid, "c" __MODULE_STRING(MAX_SSID_LEN));
MODULE_PARM_DESC(essid, "Host AP's ESSID");


/* Ethernet-II snap header */
static unsigned char snap_header[] =
{ 0xaa, 0xaa, 0x03, 0x00, 0x00, 0x00 };

static void prism2_hw_reset(struct net_device *dev);

#include "prism2_ap.c"

/* ca. 1 usec */
#define HFA384X_CMD_BUSY_TIMEOUT 100
#define HFA384X_BAP_BUSY_TIMEOUT 500

/* ca. 10 usec */
#define HFA384X_INIT_TIMEOUT 50000
#define HFA384X_CMD_COMPL_TIMEOUT 20000
#define HFA384X_ALLOC_COMPL_TIMEOUT 1000


#define TX_TIMEOUT (2 * HZ)
#define PRISM2_MAX_FRAME_SIZE 2304
#define PRISM2_MIN_MTU 256
/* FIX: */
#define PRISM2_MAX_MTU (PRISM2_MAX_FRAME_SIZE - sizeof(snap_header))


static void prism2_detach(dev_link_t *link);
static void prism2_release(u_long arg);
static int prism2_event(event_t event, int priority,
			event_callback_args_t *args);

#ifndef final_version
/* magic value written to SWSUPPORT0 reg. for detecting whether card is still
 * present */
#define HFA384X_MAGIC 0x8A32
#endif


/* event mask, i.e., events that will result in an interrupt */
/* HFA384X_EV_WTERR is mainly for development version and could be
 * removed from final version */
#if 1
#define HFA384X_EVENT_MASK \
	(HFA384X_EV_TXEXC | HFA384X_EV_TX | HFA384X_EV_RX | \
	 HFA384X_EV_WTERR | HFA384X_EV_ALLOC | HFA384X_EV_DTIM | \
	 HFA384X_EV_INFO)
#else
#define HFA384X_EVENT_MASK \
	(HFA384X_EV_RX | HFA384X_EV_ALLOC | HFA384X_EV_INFO)
#endif


#ifdef __PPC__

/* PPC */
#define HFA384X_OUTB(v,a) out_8((unsigned char *) (dev->base_addr + (a)), (v))
#define HFA384X_INB(a) in_8((unsigned char *)(dev->base_addr + (a)))
#define HFA384X_OUTW(v,a) \
	out_be16((unsigned short *) (dev->base_addr + (a)), __cpu_to_le16((v)))
#define HFA384X_INW(a) \
	__le16_to_cpu(in_be16((unsigned short *) (dev->base_addr + (a))))

#define HFA384X_OUTW_DATA(v,a) \
	out_be16((unsigned short *) (dev->base_addr + (a)), ((v)))
#define HFA384X_INW_DATA(a) \
	in_be16((unsigned short *) (dev->base_addr + (a)))

#else

/* ix86 */
#define HFA384X_OUTB(v,a) outb((v), dev->base_addr + (a))
#define HFA384X_INB(a) inb(dev->base_addr + (a))
#define HFA384X_OUTW(v,a) outw(__cpu_to_le16((v)), dev->base_addr + (a))
#define HFA384X_INW(a) __le16_to_cpu(inw(dev->base_addr + (a)))

#define HFA384X_OUTW_DATA(v,a) outw(__cpu_to_le16((v)), dev->base_addr + (a))
#define HFA384X_INW_DATA(a) inw(dev->base_addr + (a))

#endif


static u16 hfa384x_read_reg(struct net_device *dev, u16 reg)
{
	return HFA384X_INW(reg);
}


static int hfa384x_cmd(struct net_device *dev, u16 cmd, u16 param0,
		       u16 *param1, u16 *resp0)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int tries, res;
	unsigned long flags;
	u16 reg;

	spin_lock_irqsave(&local->cmdlock, flags);

	/* wait until busy bit is clear */
	tries = HFA384X_CMD_BUSY_TIMEOUT;
	while (HFA384X_INW(HFA384X_CMD_OFF) & HFA384X_CMD_BUSY && tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0) {
		reg = HFA384X_INW(HFA384X_CMD_OFF);
		spin_unlock_irqrestore(&local->cmdlock, flags);
		printk("%s: hfa384x_cmd - timeout - reg=0x%04x\n", dev->name,
		       reg);
		return -ETIMEDOUT;
	}

	/* write command */
	HFA384X_OUTW(param0, HFA384X_PARAM0_OFF);
	if (param1 != NULL)
		HFA384X_OUTW(*param1, HFA384X_PARAM1_OFF);
	HFA384X_OUTW(cmd, HFA384X_CMD_OFF);

	/* wait for command completion */
	tries = HFA384X_CMD_COMPL_TIMEOUT;
	while (!(HFA384X_INW(HFA384X_EVSTAT_OFF) & HFA384X_EV_CMD) &&
	       tries > 0) {
		tries--;
		udelay(10);
	}
	if (tries == 0) {
		reg = HFA384X_INW(HFA384X_EVSTAT_OFF);
		spin_unlock_irqrestore(&local->cmdlock, flags);
		printk("%s: hfa384x_cmd - timeout2 - reg=0x%04x\n", dev->name,
		       reg);
		return -ETIMEDOUT;
	}

	if (resp0 != NULL)
		*resp0 = HFA384X_INW(HFA384X_RESP0_OFF);
	res = (HFA384X_INW(HFA384X_STATUS_OFF) &
	       (BIT(14) | BIT(13) | BIT(12) | BIT(11) | BIT(10) | BIT(9) |
		BIT(8))) >> 8;
#ifndef final_version
	if (res) {
		u16 resp0 = HFA384X_INW(HFA384X_RESP0_OFF);
		printk("%s: CMD=0x%04x => res=0x%02x, resp0=0x%04x\n",
		       dev->name, cmd, res, resp0);
	}
#endif

	HFA384X_OUTW(HFA384X_EV_CMD, HFA384X_EVACK_OFF);

	spin_unlock_irqrestore(&local->cmdlock, flags);
	return res;
}


static int hfa384x_cmd_no_wait(struct net_device *dev, u16 cmd, u16 param0)
{
	int tries;
	u16 reg;

	/* wait until busy bit is clear */
	tries = HFA384X_CMD_BUSY_TIMEOUT;
	while (HFA384X_INW(HFA384X_CMD_OFF) & HFA384X_CMD_BUSY && tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0) {
		reg = HFA384X_INW(HFA384X_CMD_OFF);
		printk("%s: hfa384x_cmd - timeout - reg=0x%04x\n", dev->name,
		       reg);
		return -ETIMEDOUT;
	}

	/* write command */
	HFA384X_OUTW(param0, HFA384X_PARAM0_OFF);
	HFA384X_OUTW(cmd, HFA384X_CMD_OFF);

	return 0;
}


int hfa384x_setup_bap(struct net_device *dev, u16 bap, u16 id, int offset)
{
	u16 o_off, tmp;
	int tries, ret = 0;

	if (offset % 2 || bap > 1)
		return -EINVAL;

	o_off = (bap == 1) ? HFA384X_OFFSET1_OFF : HFA384X_OFFSET0_OFF;

	tries = HFA384X_BAP_BUSY_TIMEOUT;
	while (((tmp = HFA384X_INW(o_off)) & HFA384X_OFFSET_BUSY) &&
	       tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0) {
		printk("%s: hfa384x_setup_bap - timeout1\n", dev->name);
		ret = -ETIMEDOUT;
		goto out;
	}

	HFA384X_OUTW(id,
		     (bap == 1) ? HFA384X_SELECT1_OFF : HFA384X_SELECT0_OFF);

	tries = HFA384X_BAP_BUSY_TIMEOUT;
	while (((tmp = HFA384X_INW(o_off)) & HFA384X_OFFSET_BUSY) &&
	       tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0) {
		printk("%s: hfa384x_setup_bap - timeout2\n", dev->name);
		ret = -ETIMEDOUT;
		goto out;
	}

	HFA384X_OUTW(offset, o_off);

	tries = HFA384X_BAP_BUSY_TIMEOUT;
	while (((tmp = HFA384X_INW(o_off)) & HFA384X_OFFSET_BUSY) &&
	       tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0) {
		printk("%s: hfa384x_setup_bap - timeout3\n", dev->name);
		ret = -ETIMEDOUT;
		goto out;
	}
#ifndef final_version
	else if (tmp & HFA384X_OFFSET_ERR) {
		printk("%s: hfa384x_setup_bap - offset error "
		       "(%d,%d,%d,0x%04x)\n",
		       dev->name, bap, id, offset, tmp);
		ret = tmp;
	}
#endif

 out:
	return ret;
}


int hfa384x_from_bap(struct net_device *dev, u16 bap, void *buf, int len)
{
	u16 d_off;
	u16 *pos;

#ifndef final_version
	u16 o_off;
	int tries;

	o_off = (bap == 1) ? HFA384X_OFFSET1_OFF : HFA384X_OFFSET0_OFF;

	tries = HFA384X_BAP_BUSY_TIMEOUT;
	while ((HFA384X_INW(o_off) & HFA384X_OFFSET_BUSY) && tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0)
		printk(KERN_DEBUG "%s: hfa384x_from_bap(%d) - timeout\n",
		       dev->name, bap);
	else if (tries < HFA384X_BAP_BUSY_TIMEOUT)
		printk(KERN_DEBUG "%s: hfa384x_from_bap(%d) - offset was "
		       "busy\n", dev->name, bap);
#endif

	d_off = (bap == 1) ? HFA384X_DATA1_OFF : HFA384X_DATA0_OFF;

	pos = (u16 *) buf;
	for ( ; len > 1; len -= 2) {
#ifndef final_version
		if (HFA384X_INW(o_off) & HFA384X_OFFSET_BUSY)
			printk(KERN_DEBUG "%s: hfa384x_from_bap(%d) - BAP "
			       "busy during read\n", dev->name, bap);
#endif
		*pos++ = HFA384X_INW_DATA(d_off);
	}
	if (len > 0)
		*((u8 *) pos) = HFA384X_INB(d_off);

	return 0;
}


/* Offset must be even */
int hfa384x_to_bap(struct net_device *dev, u16 bap, void *buf, int len)
{
	u16 d_off;
	u16 *pos;
	int left = len;

	d_off = (bap == 1) ? HFA384X_DATA1_OFF : HFA384X_DATA0_OFF;

	pos = (u16 *) buf;
	for ( ; left > 1; left -= 2)
		HFA384X_OUTW_DATA(*pos++, d_off);
	if (left > 0)
		HFA384X_OUTB(*((char *) pos), d_off);

	return 0;
}


static int hfa384x_get_rid(struct net_device *dev, u16 rid, void *buf, int len,
			   int exact_len)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int res, rlen = 0;
	struct hfa384x_rid_hdr rec;

	res = hfa384x_cmd(dev, HFA384X_CMDCODE_ACCESS, rid, NULL, NULL);
	if (res) {
		printk("%s: hfa384x_get_rid: CMDCODE_ACCESS failed (res=%d)\n",
		       dev->name, res);
		return res;
	}

	spin_lock_bh(&local->baplock);

	res = hfa384x_setup_bap(dev, BAP0, rid, 0);
	if (!res)
		res = hfa384x_from_bap(dev, BAP0, &rec, sizeof(rec));
	if (res) {
		printk("%s: hfa384x_get_rid - from BAP0 failed\n",
		       dev->name);
		goto fail;
	}

	rlen = (__le16_to_cpu(rec.len) - 1) * 2;
	if (exact_len && rlen != len) {
		printk("RID len mismatch: rid=0x%04x, len=%d (expected %d)\n",
		       rid, rlen, len);
		res = -ENODATA;
		goto fail;
	}

	res = hfa384x_from_bap(dev, BAP0, buf, len);
	if (res) {
		printk("%s: hfa384x_get_rid - from BAP0(2) failed\n",
		       dev->name);
		goto fail;
	}

 fail:
	spin_unlock_bh(&local->baplock);

	if (res) {
		if (res == -ETIMEDOUT)
			prism2_hw_reset(dev);
		return res;
	}
	return rlen;
}


static int hfa384x_set_rid(struct net_device *dev, u16 rid, void *buf, int len)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int res;
	struct hfa384x_rid_hdr rec;

	rec.rid = __cpu_to_le16(rid);
	/* RID len in words and +1 for rec.rid */
	rec.len = __cpu_to_le16(len / 2 + len % 2 + 1);

	spin_lock_bh(&local->baplock);

	res = hfa384x_setup_bap(dev, BAP0, rid, 0);
	if (!res)
		res = hfa384x_to_bap(dev, BAP0, &rec, sizeof(rec));
	if (res) {
		printk("%s: hfa384x_set_rid - to BAP0 failed\n",
		       dev->name);
		goto fail;
	}

	res = hfa384x_to_bap(dev, BAP0, buf, len);
	if (res) {
		printk("%s: hfa384x_set_rid - to BAP0(2) failed\n", dev->name);
		goto fail;
	}

	res = hfa384x_cmd(dev, HFA384X_CMDCODE_ACCESS_WRITE, rid, NULL, NULL);
	if (res) {
		printk("%s: hfa384x_set_rid: CMDCODE_ACCESS_WRITE failed "
		       "(res=%d)\n", dev->name, res);
		goto fail;
	}

 fail:
	spin_unlock_bh(&local->baplock);

	if (res == -ETIMEDOUT)
		prism2_hw_reset(dev);

	return res;
}


static void hfa384x_disable_interrupts(struct net_device *dev)
{
	/* disable interrupts and clear event status */
	HFA384X_OUTW(0, HFA384X_INTEN_OFF);
	HFA384X_OUTW(0xffff, HFA384X_EVACK_OFF);
}


static void hfa384x_enable_interrupts(struct net_device *dev)
{
	/* ack pending events and enable interrupts from selected events */
	HFA384X_OUTW(0xffff, HFA384X_EVACK_OFF);
	HFA384X_OUTW(HFA384X_EVENT_MASK, HFA384X_INTEN_OFF);
}


static u16 hfa384x_allocate_fid(struct net_device *dev, int len)
{
	int tries;
	u16 fid;

	if (hfa384x_cmd(dev, HFA384X_CMDCODE_ALLOC, len, NULL, NULL)) {
		printk("%s: cannot allocate fid, len=%d\n", dev->name, len);
		return 0xffff;
	}

	tries = HFA384X_ALLOC_COMPL_TIMEOUT;
	while (!(HFA384X_INW(HFA384X_EVSTAT_OFF) & HFA384X_EV_ALLOC) &&
	       tries > 0) {
		tries--;
		udelay(10);
	}
	if (tries == 0) {
		printk("%s: fid allocate, len=%d - timeout\n", dev->name, len);
		return 0xffff;
	}

	fid = HFA384X_INW(HFA384X_ALLOCFID_OFF);
	HFA384X_OUTW(HFA384X_EV_ALLOC, HFA384X_EVACK_OFF);

	return fid;
}


static int prism2_reset_port(struct net_device *dev)
{
	int res;
	res = hfa384x_cmd(dev, HFA384X_CMDCODE_DISABLE, 0,
			  NULL, NULL);
	if (!res)
		res = hfa384x_cmd(dev, HFA384X_CMDCODE_ENABLE, 0,
				  NULL, NULL);

	return res;
}


int prism2_set_tim(struct net_device *dev, u16 aid, int set)
{
	u16 tmp;

	if (set) {
		PDEBUG(DEBUG_PS, "Setting TIM bit for AID %i\n", aid);
		tmp = __cpu_to_le16(0x8000 | aid);
	} else {
		PDEBUG(DEBUG_PS, "Clearing TIM bit for AID %i\n", aid);
		tmp = __cpu_to_le16(aid);
	}
	if (hfa384x_set_rid(dev, HFA384X_RID_CNFTIMCTRL, &tmp, 2)) {
		printk("   TIM AID %d setting (set=%d) failed\n", aid, set);
		return -1;
	}
	return 0;
}


static void prism2_get_version_info(struct net_device *dev, u16 rid,
				    const char *txt)
{
	struct hfa384x_comp_ident comp;

	if (hfa384x_get_rid(dev, rid, &comp, sizeof(comp), 1) < 0)
		printk("Could not get RID for component %s\n", txt);
	else
		printk("%s: %s: id=0x%02x v%d.%d.%d\n", dev->name, txt,
		       __le16_to_cpu(comp.id), __le16_to_cpu(comp.major),
		       __le16_to_cpu(comp.minor), __le16_to_cpu(comp.variant));
}


static int prism2_setup_rids(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 tmp;
	int ret = 0, len;
	char ssid[MAX_SSID_LEN + 2];

	if (local->iw_mode == IW_MODE_ADHOC && local->pseudo_adhoc)
		tmp = HFA384X_PORTTYPE_PSEUDO_IBSS;
	else if (local->iw_mode == IW_MODE_ADHOC)
		tmp = HFA384X_PORTTYPE_IBSS;
	else if (local->iw_mode == IW_MODE_INFRA)
		tmp = HFA384X_PORTTYPE_BSS;
	else
		tmp = HFA384X_PORTTYPE_HOSTAP;
	tmp = __cpu_to_le16(tmp);
	ret = hfa384x_set_rid(dev, HFA384X_RID_CNFPORTTYPE, &tmp, 2);
	if (ret) {
		printk("%s: Port type setting to %d failed\n", dev->name,
		       __le16_to_cpu(tmp));
		goto fail;
	}

	len = strlen(local->essid);
	memset(ssid, 0, sizeof(ssid));
	memcpy(ssid + 2, local->essid, len);
	ssid[0] = len;
	ret = hfa384x_set_rid(dev, HFA384X_RID_CNFOWNSSID, &ssid,
			      MAX_SSID_LEN + 2);
	if (ret) {
		printk("%s: AP own SSID setting failed\n", dev->name);
		goto fail;
	}

	tmp = __cpu_to_le16(PRISM2_DATA_MAXLEN);
	ret = hfa384x_set_rid(dev, HFA384X_RID_CNFMAXDATALEN, &tmp, 2);
	if (ret) {
		printk("%s: MAC data length setting to %d failed\n",
		       dev->name, PRISM2_DATA_MAXLEN);
		goto fail;
	}

	tmp = __cpu_to_le16(channel);
	ret = hfa384x_set_rid(dev, HFA384X_RID_CNFOWNCHANNEL, &tmp, 2);
	if (ret) {
		printk("%s: Channel setting to %d failed\n",
		       dev->name, channel);
		goto fail;
	}

	tmp = __cpu_to_le16(100);
	ret = hfa384x_set_rid(dev, HFA384X_RID_CNFBEACONINT, &tmp, 2);
	if (ret) {
		printk("%s: Beacon interval setting to %d failed\n",
		       dev->name, __le16_to_cpu(tmp));
		if (!local->is_symbol && !local->is_lucent)
			goto fail;
	}

	/* use alternate retry count 0 to make it possible to send frames
	 * without retries */
	tmp = __cpu_to_le16(0);
	ret = hfa384x_set_rid(dev, HFA384X_RID_CNFALTRETRYCOUNT, &tmp, 2);
	if (ret) {
		printk("%s: Alternate retry count setting to %d failed\n",
		       dev->name, __le16_to_cpu(tmp));
		if (!local->is_symbol && !local->is_lucent)
			goto fail;
	}

	tmp = __cpu_to_le16(HFA384X_RATES_1MBPS | HFA384X_RATES_2MBPS);
	ret = hfa384x_set_rid(dev, HFA384X_RID_CNFBASICRATES, &tmp, 2);
	if (ret) {
		printk("%s: Basic rates setting to 0x%04x failed\n",
		       dev->name, __le16_to_cpu(tmp));
		if (!local->is_symbol && !local->is_lucent)
			goto fail;
	}

	tmp = __cpu_to_le16(HFA384X_RATES_1MBPS | HFA384X_RATES_2MBPS |
			    HFA384X_RATES_5MBPS | HFA384X_RATES_11MBPS);
	ret = hfa384x_set_rid(dev, HFA384X_RID_CNFSUPPORTEDRATES, &tmp, 2);
	if (ret) {
		printk("%s: Supported rates setting to 0x%04x failed\n",
		       dev->name, __le16_to_cpu(tmp));
		if (!local->is_symbol && !local->is_lucent)
			goto fail;
	}

	tmp = __cpu_to_le16(3);
	ret = hfa384x_set_rid(dev, HFA384X_RID_CNFOWNDTIMPERIOD, &tmp, 2);
	if (ret) {
		printk("%s: DTIM period setting to %d failed\n",
		       dev->name, __le16_to_cpu(tmp));
		if (!local->is_symbol && !local->is_lucent)
			goto fail;
	}

	ret = hfa384x_set_rid(dev, HFA384X_RID_CNFDESIREDSSID, &ssid,
			      MAX_SSID_LEN + 2);
	if (ret) {
		printk("%s: Desired SSID setting failed\n", dev->name);
		goto fail;
	}

	/* Setup TXRateControl to allow use of 1, 2, 5.5, and 11 Mbps in
	 * automatic TX rate fallback */
	tmp = __cpu_to_le16(15);
	ret = hfa384x_set_rid(dev, HFA384X_RID_TXRATECONTROL, &tmp, 2);
	if (ret) {
		printk("%s: TXRateControl setting to %d failed\n",
		       dev->name, __le16_to_cpu(tmp));
		goto fail;
	}

	tmp = __cpu_to_le16(1);
	ret = hfa384x_set_rid(dev, HFA384X_RID_CREATEIBSS, &tmp, 2);
	if (ret) {
		printk("%s: Create IBSS setting to 1 failed\n", dev->name);
	}

 fail:
	return ret;
}


static int prism2_hw_config(struct net_device *dev, int initial)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int ret, i, len;

	PDEBUG(DEBUG_FLOW, "prism2_hw_config()\n");

	/* initialize HFA 384x */
	ret = hfa384x_cmd_no_wait(dev, HFA384X_CMDCODE_INIT, 0);
	if (ret) {
		printk("%s: first command failed - is the card compatible?\n",
		       dev_info);
		goto failed;
	}
	i = HFA384X_INIT_TIMEOUT;
	while (!(HFA384X_INW(HFA384X_EVSTAT_OFF) & HFA384X_EV_CMD) && i > 0) {
		i--;
		udelay(10);
	}
	if (i == 0) {
		printk("%s: card initialization timed out\n", dev_info);
		goto failed;
	}
	printk(KERN_DEBUG "prism2_hw_config: initialized in %d iterations\n",
	       HFA384X_INIT_TIMEOUT - i);
	HFA384X_OUTW(HFA384X_EV_CMD, HFA384X_EVACK_OFF);

	hfa384x_disable_interrupts(dev);

#ifndef final_version
	HFA384X_OUTW(HFA384X_MAGIC, HFA384X_SWSUPPORT0_OFF);
	if (HFA384X_INW(HFA384X_SWSUPPORT0_OFF) != HFA384X_MAGIC) {
		printk("SWSUPPORT0 write/read failed: %04X != %04X\n",
		       HFA384X_INW(HFA384X_SWSUPPORT0_OFF), HFA384X_MAGIC);
		goto failed;
	}
#endif

	/* allocate TX FIDs */
	len = local->is_symbol ? 1600 : PRISM2_TXFID_LEN;
	for (i = 0; i < PRISM2_TXFID_COUNT; i++) {
		local->txfid[i] = hfa384x_allocate_fid(dev, len);
		if (local->txfid[i] == 0xffff)
			goto failed;
		local->intransmitfid[i] = PRISM2_TXFID_EMPTY;
	}

	if (initial) {
		/* get card version information */
		prism2_get_version_info(dev, HFA384X_RID_NICID, "NIC");
		prism2_get_version_info(dev, HFA384X_RID_PRIID, "PRI");
		prism2_get_version_info(dev, HFA384X_RID_STAID, "STA");

		if (hfa384x_get_rid(dev, HFA384X_RID_CNFOWNMACADDR,
				    &dev->dev_addr, 6, 1) < 0) {
			printk("%s: could not get own MAC address\n",
			       dev->name);
		}
	}

	prism2_setup_rids(dev);
	hfa384x_enable_interrupts(dev);

	ret = hfa384x_cmd(dev, HFA384X_CMDCODE_ENABLE, 0, NULL, NULL);
	if (ret) {
		printk("%s: MAC port 0 enabling failed\n", dev->name);
		goto failed;
	}
	/* at least D-Link DWL-650 seems to require additional port reset
	 * before it starts acting as an AP, so reset port automatically
	 * here just in case */
	if (prism2_reset_port(dev)) {
		printk("%s: MAC port 0 reseting failed\n", dev->name);
		goto failed;
	}

	local->hw_ready = 1;
	local->hw_reset_tries = 0;
	return 0;

 failed:
	printk(KERN_WARNING "%s: Initialization failed\n", dev_info);
	return 1;
}


static void prism2_hw_shutdown(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;

	local->hw_ready = 0;
	if (local->link != NULL &&
	    (local->link->state & DEV_PRESENT) != DEV_PRESENT) {
		printk("%s: card already removed\n", dev_info);
		return;
	}
	if (local->link != NULL && !(local->link->state & DEV_CONFIG)) {
		printk("%s: card not configured\n", dev_info);
		return;
	}

	hfa384x_disable_interrupts(dev);

	if (hfa384x_cmd(dev, HFA384X_CMDCODE_DISABLE, 0, NULL, NULL))
		printk(KERN_WARNING "%s: Shutdown failed\n", dev_info);
}

static void prism2_cor_sreset(local_info_t *local)
{
	int res;
	conf_reg_t reg;

	reg.Function = 0;
	reg.Action = CS_READ;
	reg.Offset = CISREG_COR;
	reg.Value = 0;
	res = CardServices(AccessConfigurationRegister, local->link->handle,
			   &reg);
	if (res != CS_SUCCESS) {
		printk(KERN_DEBUG "prism2_cor_sreset failed 1 (%d)\n", res);
		return;
	}
	printk(KERN_DEBUG "prism2_cor_sreset: original COR %02x\n", reg.Value);

	reg.Action = CS_WRITE;
	reg.Value |= COR_SOFT_RESET;
	res = CardServices(AccessConfigurationRegister, local->link->handle,
			   &reg);
	if (res != CS_SUCCESS) {
		printk(KERN_DEBUG "prism2_cor_sreset failed 1 (%d)\n", res);
		return;
	}

	mdelay(1);

	reg.Value &= ~COR_SOFT_RESET;
	res = CardServices(AccessConfigurationRegister, local->link->handle,
			   &reg);
	if (res != CS_SUCCESS) {
		printk(KERN_DEBUG "prism2_cor_sreset failed 1 (%d)\n", res);
		return;
	}

	mdelay(1);
}

static void prism2_hw_reset(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int res = -1;

#if 0
	static long last_reset = 0;

	/* do not reset card more than once per second to avoid ending up in a
	 * busy loop reseting the card */
	if (last_reset + HZ >= jiffies)
		return;
	last_reset = jiffies;
#endif

	if (local->hw_resetting) {
		printk(KERN_WARNING "%s: %s: already resetting card - "
		       "ignoring reset request\n", dev_info, dev->name);
		return;
	}

	local->hw_reset_tries++;
	if (local->hw_reset_tries > 10) {
		printk(KERN_WARNING "%s: too many reset tries, skipping\n",
		       dev->name);
		return;
	}

	local->hw_resetting = 1;
	printk(KERN_WARNING "%s: %s: resetting card\n", dev_info, dev->name);
	hfa384x_disable_interrupts(dev);
	prism2_cor_sreset(local);
	if (res != 0) {
		prism2_hw_shutdown(dev);
		prism2_hw_config(dev, 0);
	}
	local->hw_resetting = 0;
}


static void cs_error(client_handle_t handle, int func, int ret)
{
	error_info_t err = { func, ret };
	CardServices(ReportError, handle, &err);
}


static struct net_device_stats *prism2_get_stats(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	return &local->stats;
}


#ifdef WIRELESS_EXT
static struct iw_statistics *prism2_get_wireless_stats(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	local->wstats.status = 0;
	local->wstats.discard.code =
		local->comm_tallies.rx_discards_wep_undecryptable;
	local->wstats.discard.misc =
		local->comm_tallies.rx_fcs_errors +
		local->comm_tallies.rx_discards_no_buffer +
		local->comm_tallies.tx_discards_wrong_sa;

	return &local->wstats;
}
#endif /* WIRELESS_EXT */


static int prism2_open(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;

	PDEBUG(DEBUG_FLOW, "prism2_open\n");
	netif_device_attach(dev);
	netif_start_queue(dev);
	local->link->open++;


	MOD_INC_USE_COUNT;

	return 0;
}


static int prism2_close(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;

	PDEBUG(DEBUG_FLOW, "prism2_close\n");
	if (!local->link->open) {
		printk("link not open?!\n");
		return 0;
	}

	local->link->open--;
	MOD_DEC_USE_COUNT;


	if (netif_running(dev)) {
		netif_stop_queue(dev);
		netif_device_detach(dev);
	} else
		if (local->link->state & DEV_STALE_CONFIG)
		mod_timer(&local->link->release, jiffies + HZ / 20);

	return 0;
}


static int prism2_change_mtu(struct net_device *dev, int new_mtu)
{
	if (new_mtu < PRISM2_MIN_MTU || new_mtu > PRISM2_MAX_MTU)
		return -EINVAL;

	dev->mtu = new_mtu;
	return 0;
}


#ifdef HAVE_TX_TIMEOUT
static void prism2_tx_timeout(struct net_device *dev)
{
	printk(KERN_WARNING "%s: %s Tx timed out! Resetting card\n",
	       dev_info, dev->name);

	prism2_hw_reset(dev);

	if (netif_queue_stopped(dev)) {
		int i;
		local_info_t *local = (local_info_t *) dev->priv;

		for (i = 0; i < PRISM2_TXFID_COUNT; i++)
			if (local->intransmitfid[i] == PRISM2_TXFID_EMPTY) {
				PDEBUG(DEBUG_EXTRA, "prism2_tx_timeout: "
				       "wake up queue\n");
				netif_wake_queue(dev);
				break;
			}
	}
}
#endif /* HAVE_TX_TIMEOUT */


int prism2_get_txfid_idx(local_info_t *local)
{
	int idx;
	unsigned long flags;

	spin_lock_irqsave(&local->txfidlock, flags);
	idx = local->next_txfid;
	do {
		if (local->intransmitfid[idx] == PRISM2_TXFID_EMPTY) {
			local->intransmitfid[idx] = PRISM2_TXFID_RESERVED;
			spin_unlock_irqrestore(&local->txfidlock, flags);
			return idx;
		}
		idx++;
		if (idx >= PRISM2_TXFID_COUNT)
			idx = 0;
	} while (idx != local->next_txfid);
	spin_unlock_irqrestore(&local->txfidlock, flags);

	PDEBUG(DEBUG_EXTRA2, "prism2_get_txfid_idx: no room in txfid buf: "
	       "packet dropped\n");
	local->stats.tx_dropped++;

	return -1;
}


/* Called only from software IRQ */
int prism2_transmit(struct net_device *dev, int idx)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int res;
	unsigned long flags;
	u16 resp0;

	spin_lock_irqsave(&local->txfidlock, flags);

	/* transmit packet */
	res = hfa384x_cmd(dev,
			  HFA384X_CMDCODE_TRANSMIT | HFA384X_CMD_TX_RECLAIM,
			  local->txfid[idx], NULL, &resp0);
	if (res) {
		spin_unlock_irqrestore(&local->txfidlock, flags);
		printk("%s: prism2_transmit: CMDCODE_TRANSMIT failed "
		       "(res=%d, resp0=0x%04x)\n", dev->name, res, resp0);
		local->stats.tx_dropped++;
		return 1;
	}
	dev->trans_start = jiffies;

	/* With reclaim, Resp0 contains new txfid for transmit; the old txfid
	 * will be automatically allocated for the next TX frame */
	local->intransmitfid[idx] = resp0;

	PDEBUG(DEBUG_FID, "prism2_transmit: [%d] txfid=0x%04x, "
	       "transmit_txfid=0x%04x\n", idx, local->txfid[idx],
	       local->intransmitfid[local->next_txfid]);

	idx++;
	if (idx >= PRISM2_TXFID_COUNT)
		idx = 0;
	local->next_txfid = idx;

	/* check if all TX buffers are occupied */
	do {
		if (local->intransmitfid[idx] == PRISM2_TXFID_EMPTY) {
			spin_unlock_irqrestore(&local->txfidlock, flags);
			return 0;
		}
		idx++;
		if (idx >= PRISM2_TXFID_COUNT)
			idx = 0;
	} while (idx != local->next_txfid);
	spin_unlock_irqrestore(&local->txfidlock, flags);

	/* no empty TX buffers, stop queue */
	netif_stop_queue(dev);

	return 0;
}


void prism2_dump_tx_header(const char *name, const struct hfa384x_tx_frame *tx)
{
	u16 fc;

	printk(KERN_DEBUG "%s: TX status=0x%04x retry_count=%d tx_rate=%d "
	       "tx_controlw=0x%04x\n",
	       name, __le16_to_cpu(tx->status), tx->retry_count, tx->tx_rate,
	       __le16_to_cpu(tx->tx_control));

	fc = __le16_to_cpu(tx->frame_control);
	printk(KERN_DEBUG "   FC type=%d:%d dur=0x%04x seq=0x%04x "
	       "data_len=%d\n",
	       WLAN_FC_GET_TYPE(fc), WLAN_FC_GET_STYPE(fc),
	       __le16_to_cpu(tx->duration_id), __le16_to_cpu(tx->seq_ctrl),
	       __le16_to_cpu(tx->data_len));

	printk(KERN_DEBUG "   A1=" MACSTR " A2=" MACSTR " A3=" MACSTR " A4="
	       MACSTR "\n",
	       MAC2STR(tx->addr1), MAC2STR(tx->addr2), MAC2STR(tx->addr3),
	       MAC2STR(tx->addr4));

	printk(KERN_DEBUG "   dst=" MACSTR " src=" MACSTR " len=%d\n",
	       MAC2STR(tx->dst_addr), MAC2STR(tx->src_addr),
	       __be16_to_cpu(tx->len));
}

/* Called only from software IRQ */
static int prism2_tx(struct sk_buff *skb, struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int len, res, idx = -1, ret = 1;
	struct hfa384x_tx_frame txdesc;
	struct sta_info *sta = NULL;
	unsigned long flags;
	u16 val;

	if ((local->link->state & (DEV_PRESENT | DEV_CONFIG)) !=
	    (DEV_PRESENT | DEV_CONFIG)) {
		printk("%s: TX, but dev not OK\n", dev->name);
		return 0;
	}

	if (!local->hw_ready) {
		printk(KERN_DEBUG "%s: prism2_tx: hw not ready - skipping\n",
		       dev->name);
		return 0;
	}

	if (skb->len < ETH_HLEN) {
		printk("%s: prism2_tx: short skb (len=%d)\n",
		       dev->name, skb->len);
		return 0;
	}

	len = skb->len;


	memset(&txdesc, 0, sizeof(txdesc));

	/* Apparently must use 802.11 header in Host AP mode */
	txdesc.tx_control = __cpu_to_le16(HFA384X_TX_CTRL_802_11 |
					  HFA384X_TX_CTRL_TX_EX |
					  HFA384X_TX_CTRL_TX_OK);

	val = (WLAN_FC_TYPE_DATA << 2) | (WLAN_FC_STYPE_DATA << 4);
	if (local->iw_mode == IW_MODE_MASTER) {
		val |= WLAN_FC_FROMDS;
		/* From DS: Addr1 = DA, Addr2 = BSSID, Addr3 = SA */
		memcpy(&txdesc.addr1, skb->data, ETH_ALEN);
		/* FIX - addr2 replaced by f/w, so no need to fill it now(?) */
		memcpy(&txdesc.addr2, dev->dev_addr, ETH_ALEN);
		memcpy(&txdesc.addr3, skb->data + 6, ETH_ALEN);
	} else if (local->iw_mode == IW_MODE_INFRA) {
		val |= WLAN_FC_TODS;
		/* To DS: Addr1 = BSSID, Addr2 = SA, Addr3 = DA;
		 * firmware sets BSSID */
		/* memcpy(&txdesc.addr1, local->bssid, ETH_ALEN); */
		memcpy(&txdesc.addr2, skb->data + 6, ETH_ALEN);
		memcpy(&txdesc.addr3, skb->data, ETH_ALEN);
	} else if (local->iw_mode == IW_MODE_ADHOC) {
		/* not From/To DS: Addr1 = DA, Addr2 = SA, Addr3 = BSSID */
		memcpy(&txdesc.addr1, skb->data, ETH_ALEN);
		memcpy(&txdesc.addr2, skb->data + 6, ETH_ALEN);
		memcpy(&txdesc.addr3, local->bssid, ETH_ALEN);
	}
	txdesc.frame_control = __cpu_to_le16(val);
	txdesc.data_len = __cpu_to_le16(skb->len - ETH_HLEN +
					sizeof(snap_header) + 2);
	memcpy(&txdesc.dst_addr, skb->data, 12);
	txdesc.len = __cpu_to_be16(skb->len - ETH_HLEN + sizeof(snap_header) +
				   2);

	if (local->iw_mode == IW_MODE_MASTER &&
	    (txdesc.addr1[0] & 0x01) == 0x00) {
		/* unicast packet - check whether destination STA is
		 * associated */
		spin_lock_irqsave(&local->ap->sta_table_lock, flags);
		sta = ap_get_sta(local->ap, txdesc.addr1);
		if (sta)
			atomic_inc(&sta->users);
		spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);

		if (!sta) {
			/* remove FromDS flag from (pseudo) ad-hoc style
			 * communication between APs */
			txdesc.frame_control &=
				~(__cpu_to_le16(WLAN_FC_FROMDS));

			printk(KERN_DEBUG "AP: packet to non-associated STA "
			       MACSTR "\n", MAC2STR(txdesc.addr1));
		}
	}

	/* Set tx_rate if using host-based TX rate control */
	if (!local->fw_tx_rate_control && sta != NULL) {
		txdesc.tx_rate = sta->tx_rate;
		sta->tx_count[sta->tx_rate_idx]++;
		sta->tx_since_last_failure++;
		if (sta->tx_since_last_failure > WLAN_RATE_UPDATE_COUNT &&
		    sta->tx_rate_idx < sta->tx_max_rate) {
			/* use next higher rate */
			while (sta->tx_rate_idx < sta->tx_max_rate) {
				sta->tx_rate_idx++;
				if (sta->tx_supp_rates &
				    (1 << sta->tx_rate_idx))
					break;
			}
			switch (sta->tx_rate_idx) {
			case 0: sta->tx_rate = 10; break;
			case 1: sta->tx_rate = 20; break;
			case 2: sta->tx_rate = 55; break;
			case 3: sta->tx_rate = 110; break;
			default: sta->tx_rate = 0; break;
			}
			PDEBUG(DEBUG_AP, "%s: STA " MACSTR " TX rate raised to"
			       " %d\n", dev->name, MAC2STR(sta->addr),
			       sta->tx_rate);
			sta->tx_since_last_failure = 0;
		}
	}

	if (sta && sta->flags & WLAN_STA_PS) {
		struct sta_buffer_frame *fbuf;
		int set_tim = 0;

		/* STA in PS mode, buffer frame for later delivery */
		if (sta->buffer_count >= STA_MAX_TX_BUFFER) {
			PDEBUG(DEBUG_PS, "No more space in this STA's PS mode "
			       "buffer\n");
			ret = 0; /* drop packet */
			local->stats.tx_dropped++;
			goto fail;
		}

		fbuf = (struct sta_buffer_frame *)
			kmalloc(sizeof(struct sta_buffer_frame), GFP_ATOMIC);
		if (fbuf == NULL) {
			printk("Malloc failed for STA's PS mode buffer\n");
			goto fail;
		}
		fbuf->next = NULL;
		fbuf->rx_time = jiffies;
		memcpy(&fbuf->txdesc, &txdesc, sizeof(txdesc));
		fbuf->skb = skb;
		spin_lock_irqsave(&local->ap->sta_table_lock, flags);
		sta->buffer_count++;
		if (sta->tx_buf_head == NULL) {
			sta->tx_buf_head = sta->tx_buf_tail = fbuf;
			set_tim = 1;
		} else {
			sta->tx_buf_tail->next = fbuf;
			sta->tx_buf_tail = fbuf;
		}
		spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);

		if (set_tim)
			prism2_set_tim(dev, sta->aid, 1);

		atomic_dec(&sta->users);

		/* do not free skb here, it will be freed when the buffered
		 * frame is sent/timed out */
		return 0;
	}

	idx = prism2_get_txfid_idx(local);
	if (idx < 0)
		goto fail;

	if (local->frame_dump & PRISM2_DUMP_TX_HDR)
		prism2_dump_tx_header(dev->name, &txdesc);

	spin_lock_bh(&local->baplock);
	res = hfa384x_setup_bap(dev, BAP0, local->txfid[idx], 0);
	if (!res)
		res = hfa384x_to_bap(dev, BAP0, &txdesc, sizeof(txdesc));
	if (!res)
		res = hfa384x_to_bap(dev, BAP0, &snap_header,
				     sizeof(snap_header));
	if (!res)
		res = hfa384x_to_bap(dev, BAP0, skb->data + 12, skb->len - 12);
	spin_unlock_bh(&local->baplock);

	if (!res)
		res = prism2_transmit(dev, idx);
	if (res) {
		printk("%s: prism2_tx - to BAP0 failed\n", dev->name);
		local->intransmitfid[idx] = PRISM2_TXFID_EMPTY;
		prism2_hw_reset(dev);
		goto fail;
	}

	add_tx_bytes(&local->stats, len);

	if (sta) {
		sta->tx_packets++;
		sta->tx_bytes += len;
		sta->last_rxtx = jiffies;
	}

	ret = 0;

 fail:
	if (sta)
		atomic_dec(&sta->users);

	if (!ret)
		dev_kfree_skb(skb);

	return ret;
}


#ifdef PRISM2_MONITOR

static int prism2_80211_header_parse(struct sk_buff *skb, unsigned char *haddr)
{
	memcpy(haddr, skb->mac.raw + 10, ETH_ALEN); /* addr2 */
	return ETH_ALEN;
}

/* Called only from hardware IRQ */
static void monitor_rx_nl(struct net_device *dev,
			  struct hfa384x_rx_frame *rxdesc)
{
	local_info_t *local = (local_info_t *) dev->priv;
	struct sk_buff *skb;
	int res;
	u16 len;

	if (local->nl_monitor == NULL) {
		printk(KERN_DEBUG "%s: monitor_rx_nl - netlink not open\n",
		       dev->name);
		return;
	}

	len = __le16_to_cpu(rxdesc->data_len);
	if (len > PRISM2_DATA_MAXLEN) {
		printk(KERN_WARNING "%s: monitor_rx_nl: len(%d) > MAX(%d)\n",
		       dev->name, len, PRISM2_DATA_MAXLEN);
		return;
	}

	skb = dev_alloc_skb(sizeof(struct hfa384x_rx_frame) + len);
	if (!skb) {
		printk(KERN_WARNING "%s: monitor_rx_nl cannot allocate "
		       "buffer\n", dev->name);
		netlink_set_err(local->nl_monitor, 0, PRISM2_MONITOR_GROUP,
				ENOBUFS);
		return;
	}

	memcpy(skb->data, rxdesc, sizeof(struct hfa384x_rx_frame));

	res = hfa384x_from_bap(dev, BAP1, skb->data +
			       sizeof(struct hfa384x_rx_frame), len);

	skb_put(skb, sizeof(struct hfa384x_rx_frame) + len);

	if (res)
		printk(KERN_WARNING "%s: monitor_rx_nl from_bap failed\n",
		       dev->name);

	skb->dev = dev;

	NETLINK_CB(skb).dst_groups = PRISM2_MONITOR_GROUP;
	netlink_broadcast(local->nl_monitor, skb, 0, PRISM2_MONITOR_GROUP,
			  GFP_ATOMIC);
}

/* Called only from hardware IRQ */
static void monitor_rx_dev(struct net_device *dev,
			   struct hfa384x_rx_frame *rxdesc)
{
	local_info_t *local = (local_info_t *) dev->priv;
	struct sk_buff *skb;
	int res, hdrlen;
	u16 len;

	hdrlen = 24; /* FIX: addr4 might be included */

	len = __le16_to_cpu(rxdesc->data_len);
	if (len > PRISM2_DATA_MAXLEN) {
		printk(KERN_WARNING "%s: monitor_rx_dev: len(%d) > MAX(%d)\n",
		       dev->name, len, PRISM2_DATA_MAXLEN);
		return;
	}

	skb = dev_alloc_skb(hdrlen + len + 2);
	if (!skb) {
		printk(KERN_WARNING "%s: monitor_rx_dev cannot allocate "
		       "buffer\n", dev->name);
		return;
	}

	/* align IP on 16b boundary */
	skb_reserve(skb, 2);
	dev->last_rx = jiffies;

	memcpy(skb->data, &rxdesc->frame_control, hdrlen);

	res = hfa384x_from_bap(dev, BAP1, skb->data + hdrlen, len);

	skb_put(skb, hdrlen + len);

	if (res)
		printk(KERN_WARNING "%s: monitor_rx_dev from_bap failed\n",
		       dev->name);

	skb->dev = dev;
	skb->mac.raw = skb->data;
	skb_pull(skb, hdrlen);
	skb->pkt_type = PACKET_OTHERHOST;
	skb->protocol = htons(ETH_P_802_2);
	netif_rx(skb);

	local->stats.rx_packets++;
	add_rx_bytes(&local->stats, hdrlen + len);
}

/* Called only from hardware IRQ */
static void monitor_rx(struct net_device *dev, struct hfa384x_rx_frame *rxdesc)
{
	local_info_t *local = (local_info_t *) dev->priv;

	if (local->monitor_type == PRISM2_MONITOR_NL)
		monitor_rx_nl(dev, rxdesc);
	else
		monitor_rx_dev(dev, rxdesc);
}

static void monitor_tx_skb(struct net_device *dev, struct sk_buff *skb)
{
	local_info_t *local;
	int idx, res;

	local = (local_info_t *) dev->priv;

	idx = prism2_get_txfid_idx(local);
	if (idx < 0)
		return;

	if (local->frame_dump & PRISM2_DUMP_TX_HDR &&
	    skb->len >= sizeof(struct hfa384x_tx_frame))
		prism2_dump_tx_header(dev->name,
				      (struct hfa384x_tx_frame *) skb->data);

	spin_lock_bh(&local->baplock);
	res = hfa384x_setup_bap(dev, BAP0, local->txfid[idx], 0);
	if (!res)
		res = hfa384x_to_bap(dev, BAP0, skb->data, skb->len);
	spin_unlock_bh(&local->baplock);

	if (!res)
		res = prism2_transmit(dev, idx);

	if (res) {
		printk("%s: monitor_tx_skb - to BAP0 failed\n", dev->name);
		local->intransmitfid[idx] = PRISM2_TXFID_EMPTY;
		if (res == -ETIMEDOUT)
			prism2_hw_reset(dev);
		return;
	}
}


static void monitor_tx(struct sock *sk, int len)
{
	struct net_device *dev;
	local_info_t *local;
	struct sk_buff *skb;

	printk("monitor_tx(%p, %d)\n", sk, len);

	if (dev_list == NULL)
		return;

	dev = dev_list->priv;
	local = (local_info_t *) dev->priv;

	if (local == NULL || local->nl_monitor == NULL)
		return;

	do {
		if (rtnl_shlock_nowait())
			return;
		while ((skb = skb_dequeue(&sk->receive_queue)) != NULL) {
			printk("monitor_tx: got skb, len=%d\n", skb->len);
			monitor_tx_skb(dev, skb);
			kfree_skb(skb);
		}
		up(&rtnl_sem);
	} while (local->nl_monitor && local->nl_monitor->receive_queue.qlen);
}
#endif /* PRISM2_MONITOR */

static void prism2_dump_rx_header(const char *name,
				  const struct hfa384x_rx_frame *rx)
{
	u16 status, fc;


	status = __le16_to_cpu(rx->status);

	printk(KERN_DEBUG "%s: RX status=0x%04x (port=%d, type=%d, "
	       "fcserr=%d) silence=%d signal=%d rate=%d rxflow=%d\n",
	       name, status, (status >> 8) & 0x07, status >> 13, status & 1,
	       rx->silence, rx->signal, rx->rate, rx->rxflow);

	fc = __le16_to_cpu(rx->frame_control);
	printk(KERN_DEBUG "   FC type=%d:%d dur=0x%04x seq=0x%04x "
	       "data_len=%d\n",
	       WLAN_FC_GET_TYPE(fc), WLAN_FC_GET_STYPE(fc),
	       __le16_to_cpu(rx->duration_id), __le16_to_cpu(rx->seq_ctrl),
	       __le16_to_cpu(rx->data_len));

	printk(KERN_DEBUG "   A1=" MACSTR " A2=" MACSTR " A3=" MACSTR " A4="
	       MACSTR "\n",
	       MAC2STR(rx->addr1), MAC2STR(rx->addr2), MAC2STR(rx->addr3),
	       MAC2STR(rx->addr4));

	printk(KERN_DEBUG "   dst=" MACSTR " src=" MACSTR " len=%d\n",
	       MAC2STR(rx->dst_addr), MAC2STR(rx->src_addr),
	       __be16_to_cpu(rx->len));
}

/* Called only from software IRQ */
static void handle_bridged_queue(void *data)
{
	local_info_t *local = (local_info_t *) data;
	struct sk_buff *skb;

	while ((skb = skb_dequeue(&local->bridge_list)) != NULL)
		dev_queue_xmit(skb);

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0))
	MOD_DEC_USE_COUNT;
#endif
}

/* Called only from hardware IRQ */
static void prism2_rx(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	struct sk_buff *skb, *skb2 = NULL;
	int res;
	u16 len = 0;
	u16 rxfid, status, macport, msg_type, fc, type, stype;
	struct hfa384x_rx_frame rxdesc;
	struct sta_info *sta = NULL;

	rxfid = HFA384X_INW(HFA384X_RXFID_OFF);
#ifndef final_version
	if (rxfid == 0) {
		rxfid = HFA384X_INW(HFA384X_RXFID_OFF);
		printk(KERN_DEBUG "prism2_rx: rxfid=0 (next 0x%04x)\n",
		       rxfid);
		prism2_hw_reset(dev);
		local->stats.rx_dropped++;
		return;
	}
#endif

	res = hfa384x_setup_bap(dev, BAP1, rxfid, 0);
	if (!res)
		res = hfa384x_from_bap(dev, BAP1, &rxdesc, sizeof(rxdesc));
	if (res) {
		printk("copy from BAP1 failed %d\n", res);
		if (res == -ETIMEDOUT)
			prism2_hw_reset(dev);
		local->stats.rx_dropped++;
		return;
	}

	if (__le16_to_cpu(rxdesc.data_len) & 0x8000) {
		/* data register seems to give 0x8000 in some error cases even
		 * though busy bit is not set in offset register; re-reading
		 * the data seems to fix at least some of the cases */
		printk(KERN_DEBUG "%s: prism2_rx: re-reading rxdesc to fix "
		       "possibly corrupted BAP read\n", dev->name);
		res = hfa384x_setup_bap(dev, BAP1, rxfid, 0);
		if (!res)
			res = hfa384x_from_bap(dev, BAP1, &rxdesc,
					       sizeof(rxdesc));
		if (res)
			printk(KERN_DEBUG "prism2_rx: could not re-read "
			       "rxdesc\n");
		if (__le16_to_cpu(rxdesc.data_len) & 0x8000)
			printk(KERN_DEBUG "prism2_rx: re-read did not fix "
			       "rxdesc\n");
	}


	status = __le16_to_cpu(rxdesc.status);
	macport = (status >> 8) & 0x07;

	if (local->frame_dump & PRISM2_DUMP_RX_HDR)
		prism2_dump_rx_header(dev->name, &rxdesc);

	if (macport != 0) {
#ifdef PRISM2_MONITOR
		if (macport == 7) {
			monitor_rx(dev, &rxdesc);
		} else {
#else
		{
#endif
			printk(KERN_DEBUG "RX: Unknown MACPort %d\n", macport);
		}
		return;
	}

	/* FCS errors should not come this far, but let's make sure that frames
	 * with errors will be dropped even in Host AP mode */
	if (status & 1) {
		printk("%s: prism2_rx: dropped FCSErr frame (status=%02X\n",
		       dev->name, status);
		return;
	}

	msg_type = status >> 13;

	if (msg_type == HFA384X_RX_MSGTYPE_MGMT) {
		if (local->iw_mode == IW_MODE_MASTER) {
			/* FIX: should STA's PS flag be also checked here? */
			hostap_rx(dev, &rxdesc, 0);
		} else {
			printk(KERN_DEBUG "%s: prism2_rx: management frame "
			       "received in non-Host AP mode\n", dev->name);
			local->stats.rx_dropped++;
			prism2_dump_rx_header(dev->name, &rxdesc);
		}
		return;
	}

	if (msg_type != HFA384X_RX_MSGTYPE_NORMAL &&
	    msg_type != HFA384X_RX_MSGTYPE_RFC1042 &&
	    msg_type != HFA384X_RX_MSGTYPE_BRIDGETUNNEL) {
		printk("%s: prism2_rx: dropped frame (msg_type=%d)\n",
		       dev->name, msg_type);
		local->stats.rx_dropped++;
		return;
	}

	fc = __le16_to_cpu(rxdesc.frame_control);
	type = WLAN_FC_GET_TYPE(fc);
	stype = WLAN_FC_GET_STYPE(fc);

	if (type != WLAN_FC_TYPE_DATA) {
		printk("%s: prism2_rx: dropped non-data frame (type=0x%02x, "
		       "subtype=0x%02x)\n", dev->name, type, stype);
		local->stats.rx_dropped++;
		return;
	}

	if (local->iw_mode == IW_MODE_MASTER && (fc & WLAN_FC_TODS)) {
		spin_lock(&local->ap->sta_table_lock);
		sta = ap_get_sta(local->ap, rxdesc.addr2);
		if (sta)
			atomic_inc(&sta->users);
		spin_unlock(&local->ap->sta_table_lock);
		if (sta == NULL || !(sta->flags & WLAN_STA_ASSOC)) {
			printk(KERN_DEBUG "%s: dropped received packet from "
			       "non-associated STA " MACSTR " (type=0x%02x, "
			       "subtype=0x%02x)\n", dev->name,
			       MAC2STR(rxdesc.addr2), type, stype);
			local->stats.rx_dropped++;
			hostap_rx(dev, &rxdesc, 1);
			if (sta)
				atomic_dec(&sta->users);
			return;
		}
	} else if (local->iw_mode == IW_MODE_MASTER) {
		if ((rxdesc.addr1[0] & 0x01) == 0x00) {
			/* FIX: should this generate a reply like the case
			 * above?
			 * with current code, STAs do not survive AP reset
			 * every time */
			printk(KERN_DEBUG "%s: dropped received packet from "
			       MACSTR " with no ToDS flag (type=0x%02x, "
			       "subtype=0x%02x)\n", dev->name,
			       MAC2STR(rxdesc.addr2), type, stype);
			prism2_dump_rx_header(dev->name, &rxdesc);
		       local->stats.rx_dropped++;
			return;
		} else {
			/* allow this for (pseudo) ad-hoc style communication
			 * between APs; originating AP removes ToDS from
			 * unicast packets, but does not currently know how to
			 * do it for broadcast packets (not easy to know,
			 * whether they are to associated STAs or
			 * not-associated APs */
			/* FIX: this accepts unwanted broadcast frames from
			 * other BSSes and these frames should be dropped if
			 * pseudo ad-hoc inter-AP communication is not needed
			 */
			printk(KERN_DEBUG "AP: allowing broadcast packet with "
			       "no ToDS\n");
		}
	}

	if (sta) {
		sta->last_rx_silence = rxdesc.silence;
		sta->last_rx_signal = rxdesc.signal;
		sta->last_rx_rate = rxdesc.rate;
		sta->last_rx_flow = rxdesc.rxflow;
		sta->last_rx_updated = 7;
		if (rxdesc.rate == 10)
			sta->rx_count[0]++;
		else if (rxdesc.rate == 20)
			sta->rx_count[1]++;
		else if (rxdesc.rate == 55)
			sta->rx_count[2]++;
		else if (rxdesc.rate == 110)
			sta->rx_count[3]++;

		if ((fc & WLAN_FC_PWRMGT) && !(sta->flags & WLAN_STA_PS)) {
			sta->flags |= WLAN_STA_PS;
			PDEBUG(DEBUG_PS, "STA " MACSTR " changed to use PS "
			       "mode (type=0x%02X, stype=0x%02X)\n",
			       MAC2STR(rxdesc.addr2), type, stype);
		} else if (!(fc & WLAN_FC_PWRMGT) &&
			   (sta->flags & WLAN_STA_PS)) {
			sta->flags &= ~WLAN_STA_PS;
			PDEBUG(DEBUG_PS, "STA " MACSTR " changed to not use "
			       "PS mode (type=0x%02X, stype=0x%02X)\n",
			       MAC2STR(rxdesc.addr2), type, stype);
			schedule_packet_send(local, sta);
		}
	}

	if (stype != WLAN_FC_STYPE_DATA &&
	    stype != WLAN_FC_STYPE_DATA_CFACK &&
	    stype != WLAN_FC_STYPE_DATA_CFPOLL &&
	    stype != WLAN_FC_STYPE_DATA_CFACKPOLL) {
#ifdef PRISM2_NULLFUNC_ACK
		if (local->iw_mode == IW_MODE_MASTER &&
		    stype == WLAN_FC_STYPE_NULLFUNC) {
			/* some STA f/w's seem to require control::ACK frame
			 * for data::nullfunc, but Prism2 f/w 0.8.0 (at least
			 * from Compaq) does not send this..
			 * maybe we should generate ACK for these frames
			 * ourselves to make power saving work with, e.g.,
			 * Lucent WaveLAN f/w */
			hostap_rx(dev, &rxdesc, 1);
			if (sta)
				atomic_dec(&sta->users);
			return;
		}
#endif
		printk(KERN_DEBUG "%s: prism2_rx: dropped data frame with no "
		       "data (type=0x%02x, subtype=0x%02x)\n",
		       dev->name, type, stype);
		local->stats.rx_dropped++;
		if (sta)
			atomic_dec(&sta->users);
		return;
	}

	len = __le16_to_cpu(rxdesc.data_len);
	if (len > PRISM2_DATA_MAXLEN) {
		printk(KERN_WARNING "%s: Rx: len(%d) > MAX(%d)\n",
		       dev->name, len, PRISM2_DATA_MAXLEN);
		local->stats.rx_dropped++;
		return;
	}

	skb = dev_alloc_skb(len + ETH_HLEN + 2);
	if (!skb) {
		printk(KERN_WARNING "%s Rx cannot allocate buffer\n",
		       dev->name);
		local->stats.rx_dropped++;
		return;
	}

	/* align IP on 16b boundary */
	skb_reserve(skb, 2);
	dev->last_rx = jiffies;

	/* receive packet (decapsulated Ethernet frame) */
	memcpy(skb->data, &rxdesc.dst_addr, ETH_HLEN);

	if (sta) {
		sta->rx_packets++;
		sta->rx_bytes += len;
		sta->last_rxtx = jiffies;
		atomic_dec(&sta->users);
	}

	if (len >= 8) {
		/* try to remove Ethernet-II snap header */
		res = hfa384x_from_bap(dev, BAP1, skb->data + ETH_HLEN, 6);
		if (res)
			goto copy_failed;

		if (memcmp(skb->data + ETH_HLEN, snap_header, 3) == 0) {
			res = hfa384x_from_bap(dev, BAP1, skb->data + 12,
					       len - 6);
			skb_put(skb, 6 + len);
		} else {
			PDEBUG(DEBUG_EXTRA,  "no SNAP? (" MACSTR ")\n",
			       MAC2STR(skb->data + ETH_HLEN));
			res = hfa384x_from_bap(dev, BAP1,
					       skb->data + ETH_HLEN + 6,
					       len - 6);
			skb_put(skb, ETH_HLEN + len);
		}
	} else {
		res = hfa384x_from_bap(dev, BAP1, skb->data + ETH_HLEN, len);
		skb_put(skb, ETH_HLEN + len);
	}
 copy_failed:
	if (res) {
		printk("%s: data copy from BAP1 failed %d\n",
		       dev->name, res);
		local->stats.rx_dropped++;
		dev_kfree_skb_irq(skb);
		return;
	}

	skb->dev = dev;

	if (local->ap->bridge_packets) {
		if (rxdesc.addr3[0] & 0x01) {
			/* copy multicast frame both to the higher layers and
			 * to the wireless media */
			skb2 = skb_clone(skb, GFP_ATOMIC);
			if (skb2 == NULL)
				printk("%s: skb_clone failed for multicast "
				       "frame\n", dev->name);
		} else {
			struct sta_info *dest_sta;
			spin_lock(&local->ap->sta_table_lock);
			dest_sta = ap_get_sta(local->ap, rxdesc.addr3);
			if (dest_sta != NULL &&
			    (dest_sta->flags & WLAN_STA_ASSOC)) {
				/* send frame directly to the associated STA
				 * using wireless media and not passing to
				 * higher layers */
				skb2 = skb;
				skb = NULL;
			}
			spin_unlock(&local->ap->sta_table_lock);
		}
	}

	if (skb2 != NULL) {
		/* send to wireless media (put in a queue and send later) */
		skb2->mac.raw = skb2->data;
		skb2->nh.raw = skb2->data + ETH_HLEN;
		skb_queue_tail(&local->bridge_list, skb2);
/* tq_scheduler was removed in 2.4.0-test12 */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
		queue_task(&local->bridge_queue, &tq_scheduler);
#else
		MOD_INC_USE_COUNT;
		if (schedule_task(&local->bridge_queue) == 0)
			MOD_DEC_USE_COUNT;
#endif

		if ((rxdesc.addr3[0] & 0x01) == 0x01)
			local->ap->bridged_multicast++;
		else
			local->ap->bridged_unicast++;
	}

	if (skb) {
		skb->protocol = eth_type_trans(skb, dev);
		netif_rx(skb);
	}

	local->stats.rx_packets++;
	add_rx_bytes(&local->stats, len);

	/* FIX: add WIRELESS_EXT & SPY handling */
}


#ifdef WIRELESS_EXT

static int prism2_get_key_idx(struct net_device *dev, int idx)
{
	u16 val;

	if (idx < 1 || idx > 4) {
		if (hfa384x_get_rid(dev,
				    HFA384X_RID_CNFWEPDEFAULTKEYID,
				    &val, 2, 1) < 0)
			return -1;
		val = __le16_to_cpu(val);
		if (val > 3) {
			printk("Invalid CNFWEPDEFAULT_KEYID %d\n", val);
			return -1;
		}

		return val;
	}

	return idx - 1;
}


static int prism2_ioctl_siwencode(struct net_device *dev, struct iw_point *erq)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int i;
	u16 val;

	if (local->iw_mode == IW_MODE_MASTER) {
		printk("%s: WEP is currently not supported in Host AP mode\n",
		       dev->name);
		return -EOPNOTSUPP;
	}

	i = prism2_get_key_idx(dev, erq->flags & IW_ENCODE_INDEX);
	if (i < 0)
		return -EINVAL;

	if (erq->pointer != (caddr_t) 0) {
		char keybuf[14];
		int keylen;

		if (erq->length > 13)
			return -EINVAL;

		memset(keybuf, 0, sizeof(keybuf));
		if (copy_from_user(keybuf, erq->pointer, erq->length))
			return -EFAULT;

		keylen = erq->length;
		if (keylen > 0 && keylen < 5)
			keylen = 6; /* first 5 octets */
		else  if (keylen > 5)
			keylen = 14; /* first 13 octets */
		if (hfa384x_set_rid(dev, HFA384X_RID_CNFDEFAULTKEY0 + i,
				    keybuf, keylen))
			return -EINVAL;
	} else {
		val = __cpu_to_le16(i);

		if (hfa384x_set_rid(dev, HFA384X_RID_CNFWEPDEFAULTKEYID, &val,
				    2) < 0) {
			printk("Setting default WEP key id failed\n");
			return -EINVAL;
		}
	}


	if (hfa384x_get_rid(dev, HFA384X_RID_CNFWEPFLAGS, &val, 2, 1) < 0) {
		printk("CNFWEPFLAGS reading failed\n");
		return -EOPNOTSUPP;
	}
	val = __le16_to_cpu(val);

	if (erq->flags & IW_ENCODE_RESTRICTED)
		val |= HFA384X_WEPFLAGS_EXCLUDEUNENCRYPTED;

	if (erq->flags & IW_ENCODE_OPEN)
		val &= ~HFA384X_WEPFLAGS_EXCLUDEUNENCRYPTED;

	if ((erq->pointer != (caddr_t) 0 && erq->length == 0) ||
	    erq->flags & IW_ENCODE_DISABLED)
		val &= ~(HFA384X_WEPFLAGS_PRIVACYINVOKED |
			 HFA384X_WEPFLAGS_EXCLUDEUNENCRYPTED);
	else
		val |= HFA384X_WEPFLAGS_PRIVACYINVOKED;

	val = __cpu_to_le16(val);
	if (hfa384x_set_rid(dev, HFA384X_RID_CNFWEPFLAGS, &val, 2)) {
		printk("CNFWEPFLAGS writing failed\n");
		return -EOPNOTSUPP;
	}

	if (prism2_reset_port(dev))
		return -EINVAL;

	return 0;
}


#ifndef IW_ENCODE_NOKEY
/* this was added in wireless.h v10 */
#define IW_ENCODE_NOKEY 0x0800
#endif


static int prism2_ioctl_giwencode(struct net_device *dev, struct iw_point *erq)
{
	int i;
	u16 val;
	char keybuf[14];
	int keylen;

	if (erq->pointer == (caddr_t) 0)
		return -EINVAL;

	i = prism2_get_key_idx(dev, erq->flags & IW_ENCODE_INDEX);
	if (i < 0)
		return -EINVAL;

	erq->flags = i + 1;

	keylen = hfa384x_get_rid(dev, HFA384X_RID_CNFDEFAULTKEY0 + i,
				 keybuf, sizeof(keybuf), 0);
	if (keylen < 0 || keylen > sizeof(keybuf)) {
		printk("WEP key reading failed\n");
		erq->flags |= IW_ENCODE_NOKEY;
	} else if (copy_to_user(erq->pointer, keybuf, keylen))
		return -EFAULT;
	else
		erq->length = keylen;

	if (hfa384x_get_rid(dev, HFA384X_RID_CNFWEPFLAGS, &val, 2, 1) < 0) {
		printk("CNFWEPFLAGS reading failed\n");
		return -EOPNOTSUPP;
	}
	val = __le16_to_cpu(val);
	if (val & HFA384X_WEPFLAGS_PRIVACYINVOKED)
		erq->flags |= IW_ENCODE_ENABLED;
	else
		erq->flags |= IW_ENCODE_DISABLED;
	if (val & HFA384X_WEPFLAGS_EXCLUDEUNENCRYPTED)
		erq->flags |= IW_ENCODE_RESTRICTED;
	else
		erq->flags |= IW_ENCODE_OPEN;

	return 0;
}


static int prism2_ioctl_giwspy(struct net_device *dev, struct iw_point *srq)
{
	local_info_t *local = (local_info_t *) dev->priv;
	struct sockaddr addr[IW_MAX_SPY];
	struct iw_quality qual[IW_MAX_SPY];

	if (srq->pointer == NULL) {
		printk("\npointer == NULL\n");
		srq->length = 0;
		return 0;
	}

	if (local->iw_mode != IW_MODE_MASTER) {
		printk("SIOCGIWSPY is currently only supported in Host AP "
		       "mode\n");
		srq->length = 0;
		return 0;
	}

	srq->length = prism2_ap_get_sta_qual(local, addr, qual);

	if (copy_to_user(srq->pointer, &addr, sizeof(addr[0]) * srq->length))
		return -EFAULT;
	if (copy_to_user(srq->pointer + sizeof(addr[0]) * srq->length, &qual,
			 sizeof(qual[0]) * srq->length))
		return -EFAULT;

	return 0;
}

#endif /* WIRELESS_EXT */


static int prism2_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd)
{
#ifdef WIRELESS_EXT
	const long freq_list[] = { 2412, 2417, 2422, 2427, 2432, 2437, 2442,
				   2447, 2452, 2457, 2462, 2467, 2472, 2484 };
#define FREQ_COUNT (sizeof(freq_list) / sizeof(freq_list[0]))
	struct iwreq *wrq = (struct iwreq *) ifr;
	struct iw_range range;
#endif
	local_info_t *local = (local_info_t *) dev->priv;
	int ret = 0, i;
	u16 val, resp0;
	struct iw_priv_args priv[] = {
		{ SIOCDEVPRIVATE, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0,
		  "inquire" },
		{ SIOCDEVPRIVATE + 1,
		  IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "ptype" },
		{ SIOCDEVPRIVATE + 2,
		  IW_PRIV_TYPE_BYTE | IW_PRIV_SIZE_FIXED | 1,
		  IW_PRIV_TYPE_BYTE | IW_PRIV_SIZE_FIXED | 1, "readmif" },
		{ SIOCDEVPRIVATE + 3,
		  IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "txratectrl" },
		{ SIOCDEVPRIVATE + 4,
		  IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "beacon_int" },
		{ SIOCDEVPRIVATE + 5,
		  IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "pseudo_ibss" }
#ifdef PRISM2_MONITOR
		,
		{ SIOCDEVPRIVATE + 6,
		  IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "monitor" }
#endif
		, { SIOCDEVPRIVATE + 7,
		    IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "reset" }
		, { SIOCDEVPRIVATE + 8,
		    IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "dump" }
	};

	switch (cmd) {

#ifdef WIRELESS_EXT
	case SIOCGIWNAME:
		strcpy(wrq->u.name, "IEEE 802.11-DS");
		break;

	case SIOCSIWFREQ:
		if (!capable(CAP_NET_ADMIN)) {
			ret = -EPERM;
			break;
		}
		/* freq => chan. */
		if (wrq->u.freq.e == 1 &&
		    wrq->u.freq.m / 100000 >= freq_list[0] &&
		    wrq->u.freq.m / 100000 <= freq_list[FREQ_COUNT - 1]) {
			int ch;
			int freq = wrq->u.freq.m / 100000;
			for (ch = 0; ch < FREQ_COUNT; ch++) {
				if (freq == freq_list[ch]) {
					wrq->u.freq.e = 0;
					wrq->u.freq.m = ch + 1;
					break;
				}
			}
		}

		/* FIX: check country-specific limitations(?) */
		if (wrq->u.freq.e != 0 || wrq->u.freq.m > FREQ_COUNT) {
			ret = -EINVAL;
			break;
		}

		val = __cpu_to_le16(wrq->u.freq.m);
		if (hfa384x_set_rid(dev, HFA384X_RID_CNFOWNCHANNEL, &val, 2) ||
		    prism2_reset_port(dev))
			ret = -EINVAL;
		break;

	case SIOCGIWFREQ:
		if (hfa384x_get_rid(dev, HFA384X_RID_CURRENTCHANNEL, &val, 2,
				    1) < 0) {
			ret = -EINVAL;
			break;
		}
		val = __le16_to_cpu(val);
		if (val < 1 || val > FREQ_COUNT) {
			ret = -EINVAL;
			break;
		}
		wrq->u.freq.m = freq_list[val - 1] * 100000;
		wrq->u.freq.e = 1;
		break;

	case SIOCGIWESSID:
		if (wrq->u.data.pointer) {
			wrq->u.data.flags = 1; /* active */
			if (local->iw_mode == IW_MODE_MASTER) {
				wrq->u.data.length = strlen(local->essid) + 1;
				copy_to_user(wrq->u.data.pointer, local->essid,
					     sizeof(local->essid));
			} else {
				int len;
				char ssid[MAX_SSID_LEN + 3];
				len = hfa384x_get_rid(dev,
						      HFA384X_RID_CURRENTSSID,
						      &ssid, MAX_SSID_LEN + 2,
						      0);
				val = __le16_to_cpu(*(u16 *) ssid);
				if (len > MAX_SSID_LEN + 2 || len < 0 ||
				    val > MAX_SSID_LEN) {
					ret = -EOPNOTSUPP;
					break;
				}
				ssid[val + 2] = '\0';
				wrq->u.data.length = val + 1;
				copy_to_user(wrq->u.data.pointer, ssid + 2,
					     val);
			}
		}
		break;

	case SIOCSIWESSID:
		if (!capable(CAP_NET_ADMIN)) {
			ret = -EPERM;
			break;
		}

		if (wrq->u.data.pointer) {
			int len;
			char ssid[MAX_SSID_LEN + 2];

			if (wrq->u.data.flags == 0) {
				local->essid[0] = '\0'; /* ANY */
			} else {
				if (wrq->u.data.length > MAX_SSID_LEN + 1) {
					ret = -E2BIG;
					break;
				}
				copy_from_user(local->essid,
					       wrq->u.data.pointer,
					       wrq->u.data.length);
				local->essid[MAX_SSID_LEN] = '\0';
			}

			len = strlen(local->essid);
			memset(ssid, 0, sizeof(ssid));
			memcpy(ssid + 2, local->essid, len);
			ssid[0] = len;
			if (hfa384x_set_rid(dev,
					    local->iw_mode == IW_MODE_INFRA ?
					    HFA384X_RID_CNFDESIREDSSID :
					    HFA384X_RID_CNFOWNSSID, &ssid,
					    MAX_SSID_LEN + 2) ||
			    prism2_reset_port(dev))
				ret = -EINVAL;

		}
		break;

	case SIOCGIWRATE:
		wrq->u.bitrate.fixed = 0;
		wrq->u.bitrate.value = 11000000;
		break;

	case SIOCGIWRTS:
		if (hfa384x_get_rid(dev, HFA384X_RID_RTSTHRESHOLD, &val, 2, 1)
		    < 0)
			ret = -EINVAL;
		else {
			wrq->u.rts.value = __le16_to_cpu(val);
			wrq->u.rts.fixed = 1;
		}
		break;

	case SIOCGIWFRAG:
		if (hfa384x_get_rid(dev, HFA384X_RID_FRAGMENTATIONTHRESHOLD,
				    &val, 2, 1) < 0)
			ret = -EINVAL;
		else {
			wrq->u.rts.value = __le16_to_cpu(val);
			wrq->u.rts.fixed = 1;
		}
		break;

	case SIOCSIWENCODE:
		if (!capable(CAP_NET_ADMIN)) { ret = -EPERM; break; }
		ret = prism2_ioctl_siwencode(dev, &wrq->u.encoding);
		break;

	case SIOCGIWENCODE:
		if (!capable(CAP_NET_ADMIN)) { ret = -EPERM; break; }
		ret = prism2_ioctl_giwencode(dev, &wrq->u.encoding);
		break;

	case SIOCGIWAP:
		if (hfa384x_get_rid(dev, HFA384X_RID_CURRENTBSSID,
				    &wrq->u.ap_addr.sa_data, ETH_ALEN,
				    ETH_ALEN) < 0) {
			ret = -EOPNOTSUPP;
			break;
		}
		/* FIX: local->bssid should be updated in a better place;
		 * maybe in info frame LinkStatus reception? */
		memcpy(local->bssid, &wrq->u.ap_addr.sa_data, ETH_ALEN);
		wrq->u.ap_addr.sa_family = ARPHRD_ETHER;

		break;

	case SIOCSIWNICKN:
		if (!capable(CAP_NET_ADMIN)) {
			ret = -EPERM;
			break;
		}

		if (wrq->u.data.pointer) {
			int len;
			char name[MAX_NAME_LEN + 3];

			if (wrq->u.data.length > MAX_NAME_LEN + 1) {
				ret = -E2BIG;
				break;
			}
			copy_from_user(&name[2], wrq->u.data.pointer,
				       wrq->u.data.length);
			name[2 + MAX_NAME_LEN] = '\0';

			len = strlen(&name[2]);
			name[0] = len;
			name[1] = 0;
			if (hfa384x_set_rid(dev, HFA384X_RID_CNFOWNNAME,
					    &name, MAX_NAME_LEN + 2))
				ret = -EINVAL;
		}
		break;

	case SIOCGIWNICKN:
		if (wrq->u.data.pointer) {
			int len;
			char name[MAX_NAME_LEN + 3];
			len = hfa384x_get_rid(dev,
					      HFA384X_RID_CNFOWNNAME,
					      &name, MAX_NAME_LEN + 2,
					      0);
			val = __le16_to_cpu(*(u16 *) name);
			if (len > MAX_NAME_LEN + 2 || len < 0 ||
			    val > MAX_NAME_LEN) {
				ret = -EOPNOTSUPP;
				break;
			}
			name[val + 2] = '\0';
			wrq->u.data.length = val + 1;
			copy_to_user(wrq->u.data.pointer, name + 2, val);
		}
		break;

	case SIOCGIWSPY:
		ret = prism2_ioctl_giwspy(dev, &wrq->u.data);
		break;

	case SIOCGIWRANGE:
		if (verify_area(VERIFY_WRITE, wrq->u.data.pointer,
				sizeof(range))) {
			ret = -EINVAL;
			break;
		}
		wrq->u.data.length = sizeof(range);
		memset(&range, 0, sizeof(range));
#if WIRELESS_EXT > 10
		range.we_version_compiled = WIRELESS_EXT;
		range.we_version_source = 11;
#endif /* WIRELESS_EXT > 10 */

		range.max_qual.qual = 0x9a;
		range.max_qual.level = 0x9a;
		range.max_qual.noise = 0x9a;

		if (copy_to_user(wrq->u.data.pointer, &range, sizeof(range))) {
			ret = -EFAULT;
			break;
		}

		break;

#if WIRELESS_EXT > 8
	case SIOCGIWMODE:
		wrq->u.mode = local->iw_mode;
		break;

	case SIOCSIWMODE:
		if (wrq->u.mode != IW_MODE_ADHOC &&
		    wrq->u.mode != IW_MODE_INFRA &&
		    wrq->u.mode != IW_MODE_MASTER) {
			ret = -EOPNOTSUPP;
			break;
		}
		if (wrq->u.mode == local->iw_mode)
			break;
		printk(KERN_DEBUG "prism2: %s: operating mode changed "
		       "%d -> %d\n", dev->name,
		       local->iw_mode, wrq->u.mode);
		local->iw_mode = wrq->u.mode;
		val = HFA384X_PORTTYPE_HOSTAP;
		if (local->iw_mode == IW_MODE_ADHOC && local->pseudo_adhoc)
			val = HFA384X_PORTTYPE_PSEUDO_IBSS;
		else if (local->iw_mode == IW_MODE_ADHOC)
			val = HFA384X_PORTTYPE_IBSS;
		else if (local->iw_mode == IW_MODE_INFRA)
			val = HFA384X_PORTTYPE_BSS;
		if (hfa384x_set_rid(dev, HFA384X_RID_CNFPORTTYPE, &val, 2)) {
			ret = -EOPNOTSUPP;
			break;
		}
		if (prism2_reset_port(dev))
			ret = -EINVAL; 
		break;
#endif

	case SIOCGIWPRIV:
		if (!wrq->u.data.pointer ||
		    verify_area(VERIFY_WRITE, wrq->u.data.pointer,
				sizeof(priv))) {
			ret = -EINVAL;
			break;
		}
		wrq->u.data.length = sizeof(priv) / sizeof(priv[0]);
		copy_to_user(wrq->u.data.pointer, priv, sizeof(priv));
		break;

	case SIOCDEVPRIVATE:
		if (!capable(CAP_NET_ADMIN)) {
			ret = -EPERM;
			break;
		}

		i = *((int *) wrq->u.name);
		if (hfa384x_cmd(dev, HFA384X_CMDCODE_INQUIRE,
				__cpu_to_le16(i), NULL, NULL))
			ret = -EOPNOTSUPP;
		break;

	case SIOCDEVPRIVATE + 1:
		if (!capable(CAP_NET_ADMIN)) {
			ret = -EPERM;
			break;
		}

		val = __cpu_to_le16(*((int *) wrq->u.name));
		if (hfa384x_set_rid(dev, HFA384X_RID_CNFPORTTYPE, &val, 2))
			ret = -EOPNOTSUPP;
		if (prism2_reset_port(dev))
			ret = -EINVAL; 
	       break;

	case SIOCDEVPRIVATE + 2:
		if (!capable(CAP_NET_ADMIN)) {
			ret = -EPERM;
			break;
		}

		val = __cpu_to_le16(*((int *) wrq->u.name));
		if (hfa384x_cmd(dev, HFA384X_CMDCODE_READMIF, val, NULL,
				&resp0))
			ret = -EOPNOTSUPP;
		else
			*(wrq->u.name) = __le16_to_cpu(resp0);
		break;

	case SIOCDEVPRIVATE + 3:
		if (!capable(CAP_NET_ADMIN)) { ret = -EPERM; break; }
		local->fw_tx_rate_control = *((int *) wrq->u.name);
		break;

	case SIOCDEVPRIVATE + 4:
		if (!capable(CAP_NET_ADMIN)) {
			ret = -EPERM;
			break;
		}

		val = __cpu_to_le16(*((int *) wrq->u.name));
		if (hfa384x_set_rid(dev, HFA384X_RID_CNFBEACONINT, &val, 2) ||
		    prism2_reset_port(dev)) {
			ret = -EINVAL;
			break;
		}

		break;

	case SIOCDEVPRIVATE + 5:
		if (!capable(CAP_NET_ADMIN)) {
			ret = -EPERM;
			break;
		}

		i = *((int *) wrq->u.name);
		if (i == local->pseudo_adhoc)
			break;
		if (i != 0 && i != 1) {
			ret = -EINVAL;
			break;
		}
		printk(KERN_DEBUG "prism2: %s: pseudo IBSS change %d -> %d\n",
		       dev->name, local->pseudo_adhoc, i);
		local->pseudo_adhoc = i;
		if (local->iw_mode != IW_MODE_ADHOC)
			break;

		if (local->pseudo_adhoc)
			val = HFA384X_PORTTYPE_PSEUDO_IBSS;
		else
			val = HFA384X_PORTTYPE_IBSS;
		if (hfa384x_set_rid(dev, HFA384X_RID_CNFPORTTYPE, &val, 2)) {
			ret = -EOPNOTSUPP;
			break;
		}
		if (prism2_reset_port(dev))
			ret = -EINVAL; 
		break;

#ifdef PRISM2_MONITOR
	case SIOCDEVPRIVATE + 6:
		if (!capable(CAP_NET_ADMIN)) {
			ret = -EPERM;
			break;
		}

		val = *((int *) wrq->u.name);
		if (val == 0) {
			struct sock *nl = local->nl_monitor;
			u16 tmp;
			printk("Disabling monitor mode\n");
			ether_setup(dev);
			local->monitor_type = PRISM2_MONITOR_OFF;
			local->nl_monitor = NULL;
			if (nl != NULL) {
				if (nl->socket == NULL)
					printk("nl->socket == NULL\n");

				/* FIX: this seems to crash the kernel, but
				 * something needs to be released.. */
				/* sock_release(nl->socket); */
			}
			hfa384x_cmd(dev, HFA384X_CMDCODE_TEST |
				    (HFA384X_TEST_STOP << 8),
				    0, NULL, NULL);
			tmp = __cpu_to_le16(HFA384X_PORTTYPE_HOSTAP);
			ret = hfa384x_set_rid(dev, HFA384X_RID_CNFPORTTYPE,
					      &tmp, 2);
			prism2_reset_port(dev);
		} else if (val == 1 || val == 2) {
			u16 tmp;
			printk("Enabling monitor mode(%i)\n", val);
			if (val == 1) {
				ether_setup(dev);
				local->monitor_type = PRISM2_MONITOR_NL;
			} else {
				dev->type = ARPHRD_IEEE80211;
				dev->hard_header_parse =
					prism2_80211_header_parse;
				local->monitor_type = PRISM2_MONITOR_DEV;
			}
			if (local->monitor_type == PRISM2_MONITOR_NL &&
			    local->nl_monitor == NULL) {
				local->nl_monitor =
					netlink_kernel_create(NETLINK_USERSOCK,
							      monitor_tx);
				if (local->nl_monitor == NULL)
					printk(KERN_WARNING
					       "netlink_kernel_create "
					       "failed\n");
			}
#ifdef PRISM2_MONITOR_PACKET_INJECT
			tmp = __cpu_to_le16(HFA384X_PORTTYPE_HOSTAP);
#else
			tmp = __cpu_to_le16(HFA384X_PORTTYPE_PSEUDO_IBSS);
#endif
			ret = hfa384x_set_rid(dev, HFA384X_RID_CNFPORTTYPE,
					      &tmp, 2);
			if (ret) {
				printk("Port type setting to pseudo IBSS "
				       "failed\n");
				ret = -EOPNOTSUPP;
				break;
			}
			tmp = __cpu_to_le16(HFA384X_WEPFLAGS_HOSTENCRYPT |
					    HFA384X_WEPFLAGS_HOSTDECRYPT);
			ret = hfa384x_set_rid(dev, HFA384X_RID_CNFWEPFLAGS,
					      &tmp, 2);
			if (ret) {
				printk("WEP flags setting failed\n");
				ret = -EOPNOTSUPP;
				break;
			}
#if 0
			tmp = __cpu_to_le16(1);
			ret = hfa384x_set_rid(dev, HFA384X_RID_PROMISCUOUSMODE,
					      &tmp, 2);
			if (ret) {
				printk("Promiscuous mode setting failed\n");
				ret = -EOPNOTSUPP;
				break;
			}
#endif
			prism2_reset_port(dev);
#ifndef PRISM2_MONITOR_PACKET_INJECT
			hfa384x_cmd(dev, HFA384X_CMDCODE_TEST |
				    (HFA384X_TEST_MONITOR << 8),
				    0, NULL, NULL);
#endif
		} else
			ret = -EINVAL;
		break;
#endif /* PRISM2_MONITOR */

	case SIOCDEVPRIVATE + 7:
	{
		if (!capable(CAP_NET_ADMIN)) {
			ret = -EPERM;
			break;
		}

		val = *((int *) wrq->u.name);

		printk("%s: manual reset request(%d)\n", dev->name, val);
		if (val)
			prism2_hw_reset(dev);
		else {
			prism2_hw_shutdown(dev);
			prism2_hw_config(dev, 0);
		}
		break;
	}

	case SIOCDEVPRIVATE + 8:
		if (!capable(CAP_NET_ADMIN)) {
			ret = -EPERM;
			break;
		}

		i = *((int *) wrq->u.name);
		if (i == local->frame_dump)
			break;
		printk(KERN_DEBUG "prism2: %s: frame dump change 0x%02x -> "
		       "0x%02x\n", dev->name, local->frame_dump, i);
		local->frame_dump = i;
		break;

	/* not supported wireless extensions */
	case SIOCSIWNAME:
	case SIOCSIWNWID:
	case SIOCGIWNWID:
		ret = -EOPNOTSUPP;
		break;

	/* FIX: add support for (at least some of) these: */
#if WIRELESS_EXT > 8
	case SIOCSIWPOWER:
	case SIOCGIWPOWER:
#endif
#if WIRELESS_EXT > 9
	case SIOCSIWTXPOW:
	case SIOCGIWTXPOW:
#endif
#if WIRELESS_EXT > 10
	case SIOCSIWRETRY:
	case SIOCGIWRETRY:
#endif
	case SIOCSIWSENS:
	case SIOCGIWSENS:
	case SIOCSIWRANGE:
	case SIOCSIWPRIV:
	case SIOCSIWSPY:
	case SIOCSIWAP:
	case SIOCGIWAPLIST:
	case SIOCSIWRATE:
	case SIOCSIWRTS:
	case SIOCSIWFRAG:
		printk(KERN_DEBUG "%s: %s unsupported WIRELESS_EXT "
		       "ioctl(0x%04x)\n", dev_info, dev->name, cmd);
		ret = -EOPNOTSUPP;
		break;

#endif

		/* add: SIOCDEVPRIVATE + 0x2, + 0x3, + 0xd, + 0xe; own debug
		 * ioctl's? */

	default:
		printk("%s: %s unsupported ioctl(0x%04x)\n", dev_info,
		       dev->name, cmd);
		ret = -EOPNOTSUPP;
		break;
	}

	return ret;
}


static int prism2_debug_proc_read(char *page, char **start, off_t off,
				  int count, int *eof, void *data)
{
	char *p = page;
	local_info_t *local = (local_info_t *) data;
	int i;
	unsigned long flags;

	if (off != 0) {
		*eof = 1;
		return 0;
	}

	spin_lock_irqsave(&local->txfidlock, flags);
	p += sprintf(p, "next_txfid=%d next_alloc=%d\n",
		     local->next_txfid, local->next_alloc);
	for (i = 0; i < PRISM2_TXFID_COUNT; i++)
		p += sprintf(p, "FID: tx=%04X intransmit=%04X\n",
			     local->txfid[i], local->intransmitfid[i]);
	spin_unlock_irqrestore(&local->txfidlock, flags);
	p += sprintf(p, "FW TX rate control: %d\n", local->fw_tx_rate_control);

	return (p - page);
}


static int prism2_stats_proc_read(char *page, char **start, off_t off,
				  int count, int *eof, void *data)
{
	char *p = page;
	local_info_t *local = (local_info_t *) data;
	struct comm_tallies_sums *sums = (struct comm_tallies_sums *)
		&local->comm_tallies;

	if (off != 0) {
		*eof = 1;
		return 0;
	}

	p += sprintf(p, "TxUnicastFrames=%u\n", sums->tx_unicast_frames);
	p += sprintf(p, "TxMulticastframes=%u\n", sums->tx_multicast_frames);
	p += sprintf(p, "TxFragments=%u\n", sums->tx_fragments);
	p += sprintf(p, "TxUnicastOctets=%u\n", sums->tx_unicast_octets);
	p += sprintf(p, "TxMulticastOctets=%u\n", sums->tx_multicast_octets);
	p += sprintf(p, "TxDeferredTransmissions=%u\n",
		     sums->tx_deferred_transmissions);
	p += sprintf(p, "TxSingleRetryFrames=%u\n",
		     sums->tx_single_retry_frames);
	p += sprintf(p, "TxMultipleRetryFrames=%u\n",
		     sums->tx_multiple_retry_frames);
	p += sprintf(p, "TxRetryLimitExceeded=%u\n",
		     sums->tx_retry_limit_exceeded);
	p += sprintf(p, "TxDiscards=%u\n", sums->tx_discards);
	p += sprintf(p, "RxUnicastFrames=%u\n", sums->rx_unicast_frames);
	p += sprintf(p, "RxMulticastFrames=%u\n", sums->rx_multicast_frames);
	p += sprintf(p, "RxFragments=%u\n", sums->rx_fragments);
	p += sprintf(p, "RxUnicastOctets=%u\n", sums->rx_unicast_octets);
	p += sprintf(p, "RxMulticastOctets=%u\n", sums->rx_multicast_octets);
	p += sprintf(p, "RxFCSErrors=%u\n", sums->rx_fcs_errors);
	p += sprintf(p, "RxDiscardsNoBuffer=%u\n",
		     sums->rx_discards_no_buffer);
	p += sprintf(p, "TxDiscardsWrongSA=%u\n", sums->tx_discards_wrong_sa);
	p += sprintf(p, "RxDiscardsWEPUndecryptable=%u\n",
		     sums->rx_discards_wep_undecryptable);
	p += sprintf(p, "RxMessageInMsgFragments=%u\n",
		     sums->rx_message_in_msg_fragments);
	p += sprintf(p, "RxMessageInBadMsgFragments=%u\n",
		     sums->rx_message_in_bad_msg_fragments);
	/* FIX: this may grow too long for one page(?) */

	return (p - page);
}


static int prism2_registers_proc_read(char *page, char **start, off_t off,
				      int count, int *eof, void *data)
{
	char *p = page;
	local_info_t *local = (local_info_t *) data;

	if (off != 0) {
		*eof = 1;
		return 0;
	}

#define SHOW_REG(n) \
p += sprintf(p, "%s=%04x\n", #n, \
hfa384x_read_reg(local->dev, HFA384X_##n##_OFF))

	SHOW_REG(CMD);
	SHOW_REG(PARAM0);
	SHOW_REG(PARAM1);
	SHOW_REG(PARAM2);
	SHOW_REG(STATUS);
	SHOW_REG(RESP0);
	SHOW_REG(RESP1);
	SHOW_REG(RESP2);
	SHOW_REG(INFOFID);
	SHOW_REG(CONTROL);
	SHOW_REG(SELECT0);
	SHOW_REG(SELECT1);
	SHOW_REG(OFFSET0);
	SHOW_REG(OFFSET1);
	SHOW_REG(RXFID);
	SHOW_REG(ALLOCFID);
	SHOW_REG(TXCOMPLFID);
	SHOW_REG(SWSUPPORT0);
	SHOW_REG(SWSUPPORT1);
	SHOW_REG(SWSUPPORT2);
	SHOW_REG(EVSTAT);
	SHOW_REG(INTEN);
	SHOW_REG(EVACK);
	/* Do not read data registers, because they change the state of the
	 * MAC (offset += 2) */
	/* SHOW_REG(DATA0); */
	/* SHOW_REG(DATA1); */
	SHOW_REG(AUXPAGE);
	SHOW_REG(AUXOFFSET);
	/* SHOW_REG(AUXDATA); */

	return (p - page);
}

#define RID(n,t) { HFA384X_RID_##n, #n, t }
enum { RID_HEXDUMP, RID_WORD, RID_HWADDR, RID_STRING, RID_COMPID,
       RID_SUPRANGE };

struct {
	u16 rid;
	char *name;
	int type;
} rid_table[] = {
	RID(CNFPORTTYPE, RID_WORD),
	RID(CNFOWNMACADDR, RID_HWADDR),
	RID(CNFDESIREDSSID, RID_STRING),
	RID(CNFOWNCHANNEL, RID_WORD),
	RID(CNFOWNSSID, RID_STRING),
	RID(CNFOWNATIMWINDOW, RID_WORD),
	RID(CNFSYSTEMSCALE, RID_WORD),
	RID(CNFMAXDATALEN, RID_WORD),
	RID(CNFWDSADDRESS, RID_HWADDR),
	RID(CNFPMENABLED, RID_WORD),
	RID(CNFPMEPS, RID_WORD),
	RID(CNFMULTICASTRECEIVE, RID_WORD),
	RID(CNFMAXSLEEPDURATION, RID_WORD),
	RID(CNFPMHOLDOVERDURATION, RID_WORD),
	RID(CNFOWNNAME, RID_STRING),
	RID(CNFOWNDTIMPERIOD, RID_WORD),
	RID(CNFWDSADDRESS1, RID_HWADDR),
	RID(CNFWDSADDRESS2, RID_HWADDR),
	RID(CNFWDSADDRESS3, RID_HWADDR),
	RID(CNFWDSADDRESS4, RID_HWADDR),
	RID(CNFWDSADDRESS5, RID_HWADDR),
	RID(CNFWDSADDRESS6, RID_HWADDR),
	RID(CNFMULTICASTPMBUFFERING, RID_WORD),
	RID(CNFWEPDEFAULTKEYID, RID_WORD),
	RID(CNFDEFAULTKEY0, RID_HEXDUMP),
	RID(CNFDEFAULTKEY1, RID_HEXDUMP),
	RID(CNFDEFAULTKEY2, RID_HEXDUMP),
	RID(CNFDEFAULTKEY3, RID_HEXDUMP),
	RID(CNFWEPFLAGS, RID_HEXDUMP),
	RID(CNFWEPKEYMAPPINGTABLE, RID_HEXDUMP),
	RID(CNFAUTHENTICATION, RID_WORD),
	RID(CNFMAXASSOCSTA, RID_WORD),
	RID(CNFTXCONTROL, RID_WORD),
	RID(CNFROAMINGMODE, RID_WORD),
	RID(CNFHOSTAUTHENTICATION, RID_WORD),
	RID(CNFRCVCRCERROR, RID_WORD),
	RID(CNFMMLIFE, RID_WORD),
	RID(CNFALTRETRYCOUNT, RID_WORD),
	RID(CNFBEACONINT, RID_WORD),
	RID(CNFAPPCFINFO, RID_HEXDUMP),
	RID(CNFSTAPCFINFO, RID_HEXDUMP),
	RID(CNFPRIORITYQUSAGE, RID_HEXDUMP),
	RID(CNFTIMCTRL, RID_WORD),
	RID(CNFTHIRTY2TALLY, RID_WORD),
	RID(CNFENHSECURITY, RID_WORD),
	RID(GROUPADDRESSES, RID_HEXDUMP),
	RID(CREATEIBSS, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD, RID_WORD),
	RID(RTSTHRESHOLD, RID_WORD),
	RID(TXRATECONTROL, RID_WORD),
	RID(PROMISCUOUSMODE, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD0, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD1, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD2, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD3, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD4, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD5, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD6, RID_WORD),
	RID(RTSTHRESHOLD0, RID_WORD),
	RID(RTSTHRESHOLD1, RID_WORD),
	RID(RTSTHRESHOLD2, RID_WORD),
	RID(RTSTHRESHOLD3, RID_WORD),
	RID(RTSTHRESHOLD4, RID_WORD),
	RID(RTSTHRESHOLD5, RID_WORD),
	RID(RTSTHRESHOLD6, RID_WORD),
	RID(CNFSHORTPREAMBLE, RID_WORD),
	RID(CNFEXCLUDELONGPREAMBLE, RID_WORD),
	RID(CNFAUTHENTICATIONRSPTO, RID_WORD),
	RID(CNFBASICRATES, RID_HEXDUMP),
	RID(CNFSUPPORTEDRATES, RID_HEXDUMP),
	RID(TICKTIME, RID_WORD),
	RID(SCANREQUEST, RID_HEXDUMP),
	RID(JOINREQUEST, RID_HEXDUMP),
	RID(AUTHENTICATESTATION, RID_HEXDUMP),
	RID(CHANNELINFOREQUEST, RID_HEXDUMP),

	RID(MAXLOADTIME, RID_WORD),
	RID(DOWNLOADBUFFER, RID_HEXDUMP),
	RID(PRIID, RID_COMPID),
	RID(PRISUPRANGE, RID_SUPRANGE),
	RID(CFIACTRANGES, RID_SUPRANGE),
	RID(NICSERNUM, RID_STRING),
	RID(NICID, RID_COMPID),
	RID(MFISUPRANGE, RID_SUPRANGE),
	RID(CFISUPRANGE, RID_SUPRANGE),
	RID(CHANNELLIST, RID_HEXDUMP),
	RID(REGULATORYDOMAINS, RID_STRING),
	RID(TEMPTYPE, RID_WORD),
	RID(CIS, RID_HEXDUMP),
	RID(STAID, RID_COMPID),
	RID(STASUPRANGE, RID_SUPRANGE),
	RID(MFIACTRANGES, RID_SUPRANGE),
	RID(CFIACTRANGES2, RID_SUPRANGE),
	RID(PORTSTATUS, RID_WORD),
	RID(CURRENTSSID, RID_STRING),
	RID(CURRENTBSSID, RID_HWADDR),
	RID(COMMSQUALITY, RID_HEXDUMP),
	RID(CURRENTTXRATE, RID_WORD),
	RID(CURRENTBEACONINTERVAL, RID_WORD),
	RID(PROTOCOLRSPTIME, RID_WORD),
	RID(SHORTRETRYLIMIT, RID_WORD),
	RID(LONGRETRYLIMIT, RID_WORD),
	RID(MAXTRANSMITLIFETIME, RID_WORD),
	RID(MAXRECEIVELIFETIME, RID_WORD),
	RID(CFPOLLABLE, RID_WORD),
	RID(AUTHENTICATIONALGORITHMS, RID_HEXDUMP),
	RID(PRIVACYOPTIONIMPLEMENTED, RID_WORD),
	RID(CURRENTTXRATE1, RID_WORD),
	RID(CURRENTTXRATE2, RID_WORD),
	RID(CURRENTTXRATE3, RID_WORD),
	RID(CURRENTTXRATE4, RID_WORD),
	RID(CURRENTTXRATE5, RID_WORD),
	RID(CURRENTTXRATE6, RID_WORD),
	RID(OWNMACADDR, RID_HWADDR),
	RID(SCANRESULTSTABLE, RID_HEXDUMP),
	RID(PHYTYPE, RID_WORD),
	RID(CURRENTCHANNEL, RID_WORD),
	RID(CURRENTPOWERSTATE, RID_WORD),
	RID(CCAMODE, RID_WORD),
	RID(SUPPORTEDDATARATES, RID_HEXDUMP),

	RID(BUILDSEQ, RID_HEXDUMP),
	RID(FWID, RID_STRING)
};

#define PROC_LIMIT (PAGE_SIZE - 80)

static int prism2_rids_proc_read(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	char *p = page;
	local_info_t *local = (local_info_t *) data;
	int i, j, len, strlen, total = 0;
	unsigned char buf[256];
	struct hfa384x_comp_ident *compid;
	struct hfa384x_sup_range *range;

	for (i = 0; i < sizeof(rid_table) / sizeof(rid_table[0]); i++) {
		if (total + (p - page) <= off) {
			total += p - page;
			p = page;
		}
		if (total + (p - page) > off + count)
			break;
		if ((p - page) > PROC_LIMIT)
			break;

		p += sprintf(p, "%s=", rid_table[i].name);
		len = hfa384x_get_rid(local->dev, rid_table[i].rid, buf,
				      sizeof(buf), 0);
		if (len < 0) {
			p += sprintf(p, "<RID READ ERROR %d>\n", len);
			continue;
		}

		switch (rid_table[i].type) {
		case RID_HEXDUMP:
			for (j = 0; j < len; j++)
				p += sprintf(p, "<%02x>", buf[j]);
			p += sprintf(p, "\n");
			break;

		case RID_WORD:
			if (len != 2) {
				p += sprintf(p, "<INVALID RID_WORD LEN %d>\n",
					     len);
			} else {
				u16 val = __le16_to_cpu(*(u16 *)buf);
				p += sprintf(p, "%d\n", val);
			}
			break;

		case RID_HWADDR:
			if (len != 6) {
				p += sprintf(p,
					     "<INVALID RID_HWADDR LEN %d>\n",
					     len);
			} else {
				p += sprintf(p, MACSTR "\n", MAC2STR(buf));
			}
			break;

		case RID_STRING:
			strlen = __le16_to_cpu(*(u16 *)buf);
			if (strlen > len)
				strlen = len;
			for (j = 2; j < strlen + 2; j++) {
				if (buf[j] >= 32 && buf[j] < 127)
					p += sprintf(p, "%c", buf[j]);
				else
					p += sprintf(p, "<%02x>", buf[j]);
			}
			p += sprintf(p, "\n");
			break;

		case RID_COMPID:
			if (len != sizeof(*compid)) {
				p += sprintf(p, "<INVALID RID_COMPID LEN "
					     "%d>\n", len);
				break;
			}
			compid = (struct hfa384x_comp_ident *) buf;
			p += sprintf(p, "0x%02x v%d.%d.%d\n",
				     __le16_to_cpu(compid->id),
				     __le16_to_cpu(compid->major),
				     __le16_to_cpu(compid->minor),
				     __le16_to_cpu(compid->variant));
			break;

		case RID_SUPRANGE:
			if (len != sizeof(*range)) {
				p += sprintf(p, "<INVALID RID_SUPRANGE LEN "
					     "%d>\n", len);
				break;
			}
			range = (struct hfa384x_sup_range *) buf;
			p += sprintf(p, "%d 0x%02x %d %d-%d\n",
				     __le16_to_cpu(range->role),
				     __le16_to_cpu(range->id),
				     __le16_to_cpu(range->variant),
				     __le16_to_cpu(range->bottom),
				     __le16_to_cpu(range->top));
			break;

		default:
			p += sprintf(p, "<UNKNOWN TYPE %d>\n",
				     rid_table[i].type);
			break;
		}
	}

	total += (p - page);
	if (total >= off + count)
		*eof = 1;

	if (total < off) {
		*eof = 1;
		return 0;
	}

	len = total - off;
	if (len > (p - page))
		len = p - page;
	*start = p - len;
	if (len > count)
		len = count;

	return len;
}


static void prism2_init_proc(local_info_t *local)
{
	local->proc = NULL;

	if (proc_net != NULL) {
		if (prism2_proc == NULL)
			prism2_proc = proc_mkdir("prism2", proc_net);
		if (prism2_proc != NULL)
			local->proc = proc_mkdir(local->dev->name,
						 prism2_proc);
	}

	if (local->proc == NULL) {
		printk(KERN_INFO "/proc/net/prism2 creation failed\n");
		return;
	}

	create_proc_read_entry("debug", 0, local->proc,
			       prism2_debug_proc_read, local);
	create_proc_read_entry("stats", 0, local->proc,
			       prism2_stats_proc_read, local);
	create_proc_read_entry("registers", 0, local->proc,
			       prism2_registers_proc_read, local);
	create_proc_read_entry("rids", 0, local->proc,
			       prism2_rids_proc_read, local);
}


static void prism2_remove_proc(local_info_t *local)
{
	if (local->proc != NULL) {
		remove_proc_entry("rids", local->proc);
		remove_proc_entry("registers", local->proc);
		remove_proc_entry("stats", local->proc);
		remove_proc_entry("debug", local->proc);
		if (local->dev != NULL && local->dev->name != NULL)
			remove_proc_entry(local->dev->name, prism2_proc);
	}
}


/* allocate local data and register with CardServices
 * initialize dev_link structure, but do not configure the card yet */
static dev_link_t *prism2_attach(void)
{
	dev_link_t *link;
	local_info_t *local;
	struct net_device *dev;
	client_reg_t client_reg;
	int ret;

	for (link = dev_list; link; link = link->next) {
		if (link->state & DEV_STALE_LINK) {
			printk("%s: flushing stale link\n", dev_info);
			prism2_detach(link);
		}
	}

	link = kmalloc(sizeof(dev_link_t), GFP_KERNEL);
	local = kmalloc(sizeof(local_info_t), GFP_KERNEL);
	if (local) {
		memset(local, 0, sizeof(local_info_t));
		local->ap = kmalloc(sizeof(struct ap_data), GFP_KERNEL);
		if (local->ap)
			memset(local->ap, 0, sizeof(struct ap_data));
	}
	dev = kmalloc(sizeof(struct net_device), GFP_KERNEL);
	if (link == NULL || local == NULL || local->ap == NULL || dev == NULL)
	{
		printk(KERN_WARNING "prism2_attach - could not kmalloc memory"
		       "\n");
		if (link) kfree(link);
		if (local) {
			if (local->ap) kfree(local->ap);
			kfree(local);
		}
		if (dev) kfree(dev);
		return NULL;
	}
	memset(link, 0, sizeof(dev_link_t));
	memset(dev, 0, sizeof(struct net_device));

	link->priv = dev;
	dev->priv = local;
	local->link = link;
	local->dev = dev;

	spin_lock_init(&local->txfidlock);
	spin_lock_init(&local->cmdlock);
	spin_lock_init(&local->baplock);

	memcpy(local->essid, essid, sizeof(local->essid));
#ifdef WIRELESS_EXT
	local->iw_mode = IW_MODE_MASTER;
#endif

	skb_queue_head_init(&local->bridge_list);

	/* Initialize task queue structure for bridged packet handling */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
	local->bridge_queue.next = NULL;
#else
	INIT_LIST_HEAD(&local->bridge_queue.list);
#endif
	local->bridge_queue.sync = 0;
	local->bridge_queue.routine = (void (*)(void *))handle_bridged_queue;
	local->bridge_queue.data = local;

	ether_setup(dev);

	/* kernel callbacks */
	dev->get_stats = prism2_get_stats;
#ifdef WIRELESS_EXT
	dev->get_wireless_stats = prism2_get_wireless_stats;
#endif
	dev->open = prism2_open;
	dev->stop = prism2_close;
	dev->hard_start_xmit = prism2_tx;
#ifdef HAVE_MULTICAST
	/* FIX: to be implemented as soon as Prism2 supports GroupAddresses
	 * and correct documentation is available */

	/* dev->set_multicast_list = prism2_set_multicast_list; */
#endif
#ifdef HAVE_PRIVATE_IOCTL
	dev->do_ioctl = prism2_ioctl;
#endif
#ifdef HAVE_CHANGE_MTU
	dev->change_mtu = prism2_change_mtu;
#endif
#ifdef HAVE_TX_TIMEOUT
	dev->tx_timeout = prism2_tx_timeout;
	dev->watchdog_timeo = TX_TIMEOUT;
#endif

	init_dev_name(dev, local->node);
	dev->mtu = mtu;
	netif_stop_queue(dev);

	link->release.function = &prism2_release;
	link->release.data = (u_long)link;

	PDEBUG(DEBUG_HW, "%s: setting Vcc=33 (constant)\n", dev_info);
	link->conf.Vcc = 33;
	link->conf.IntType = INT_MEMORY_AND_IO;

	/* register with CardServices */
	link->next = dev_list;
	dev_list = link;
	client_reg.dev_info = &dev_info;
	client_reg.Attributes = INFO_IO_CLIENT;
	client_reg.EventMask = CS_EVENT_CARD_INSERTION |
		CS_EVENT_CARD_REMOVAL |
		CS_EVENT_RESET_PHYSICAL | CS_EVENT_CARD_RESET |
		CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
	client_reg.event_handler = &prism2_event;
	client_reg.Version = 0x0210;
	client_reg.event_callback_args.client_data = link;
	ret = CardServices(RegisterClient, &link->handle, &client_reg);
	if (ret != CS_SUCCESS) {
		cs_error(link->handle, RegisterClient, ret);
		prism2_detach(link);
		return NULL;
	}
	return link;
}


static void prism2_detach(dev_link_t *link)
{
	dev_link_t **linkp;

	PDEBUG(DEBUG_FLOW, "prism2_detach\n");

	for (linkp = &dev_list; *linkp; linkp = &(*linkp)->next)
		if (*linkp == link)
			break;
	if (*linkp == NULL) {
		printk(KERN_WARNING "%s: Attempt to detach non-existing "
		       "PCMCIA client\n", dev_info);
		return;
	}

	del_timer(&link->release);
	if (link->state & DEV_CONFIG) {
		printk("%s: detach postponed, '%s' still locked\n",
		       dev_info, link->dev->dev_name);
		prism2_release((u_long)link);
		if (link->state & DEV_STALE_CONFIG) {
			link->state |= DEV_STALE_LINK;
			return;
		}
	}

	if (link->handle) {
		int res = CardServices(DeregisterClient, link->handle);
		if (res) {
			printk("CardService(DeregisterClient) => %d\n", res);
			cs_error(link->handle, DeregisterClient, res);
		}
	}

	*linkp = link->next;
	/* release local_info_t struct */
	if (link->priv) {
		struct net_device *dev = (struct net_device *) link->priv;
		local_info_t *local = (local_info_t *) dev->priv;
		if (local->ap != NULL)
			ap_free_data(local->ap);
		prism2_remove_proc(local);
		if (link->dev) {
			unregister_netdev(dev);
			printk("%s: Netdevice unregistered\n", dev_info);
		} else
			printk("%s: link->dev == NULL: do not unregister "
			       "netdevice\n", dev_info);
#ifdef PRISM2_MONITOR
		if (local->nl_monitor != NULL) {
			if (local->nl_monitor->socket == NULL)
				printk("nl_monitor->socket == NULL\n");
			/* this seems to crash the kernel.. */
			/* sock_release(local->nl_monitor->socket); */
		}
#endif
		kfree(local);
		kfree(dev);
	}
	kfree(link);
}


/* Called only from hardware IRQ */
static void prism2_alloc_ev(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int idx;
	u16 fid;

	fid = HFA384X_INW(HFA384X_ALLOCFID_OFF);

	PDEBUG(DEBUG_FID, "FID: interrupt: ALLOC - fid=0x%04x\n", fid);

	idx = local->next_alloc;

	spin_lock(&local->txfidlock);

	do {
		if (local->txfid[idx] == fid) {
			PDEBUG(DEBUG_FID, "FID: found matching txfid[%d]\n",
			       idx);
#ifndef final_version
			if (local->intransmitfid[idx] == PRISM2_TXFID_EMPTY)
				printk("Already released txfid found at idx "
				       "%d\n", idx);
			if (local->intransmitfid[idx] == PRISM2_TXFID_RESERVED)
				printk("Already reserved txfid found at idx "
				       "%d\n", idx);
#endif
			local->intransmitfid[idx] = PRISM2_TXFID_EMPTY;
			idx++;
			local->next_alloc = idx >= PRISM2_TXFID_COUNT ? 0 :
				idx;

			if (netif_queue_stopped(dev))
				netif_wake_queue(dev);

			goto out;
		}

		idx++;
		if (idx >= PRISM2_TXFID_COUNT)
			idx = 0;
	} while (idx != local->next_alloc);

	printk(KERN_WARNING "%s: could not found matching txfid (0x%04x) for "
	       "alloc event\n", dev->name, HFA384X_INW(HFA384X_ALLOCFID_OFF));

 out:
	spin_unlock(&local->txfidlock);
}


/* Called only from hardware IRQ */
static void prism2_txexc(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 fid, status, fc;
	int res;
	struct hfa384x_tx_frame txdesc;
	struct sta_info *sta;

	local->stats.tx_errors++;
	fid = HFA384X_INW(HFA384X_TXCOMPLFID_OFF);
	PDEBUG(DEBUG_EXTRA, "%s: TXEXC - fid=0x%04x - ", dev->name, fid);

	res = hfa384x_setup_bap(dev, BAP1, fid, 0);
	if (!res)
		res = hfa384x_from_bap(dev, BAP1, &txdesc, sizeof(txdesc));
	if (res) {
		PDEBUG2(DEBUG_EXTRA, "could not get TXEXC txframe\n");
		if (res == -ETIMEDOUT)
			prism2_hw_reset(dev);
		return;
	}
	status = __le16_to_cpu(txdesc.status);
	PDEBUG2(DEBUG_EXTRA, "status=0x%04x (%s%s%s%s) tx_control=%04x\n",
		status,
		status & HFA384X_TX_STATUS_RETRYERR ? "[RetryErr]" : "",
		status & HFA384X_TX_STATUS_AGEDERR ? "[AgedErr]" : "",
		status & HFA384X_TX_STATUS_DISCON ? "[Discon]" : "",
		status & HFA384X_TX_STATUS_FORMERR ? "[FormErr]" : "",
		__le16_to_cpu(txdesc.tx_control));
	fc = __le16_to_cpu(txdesc.frame_control);
	PDEBUG(DEBUG_EXTRA, "   retry_count=%d tx_rate=%d fc=0x%04x "
	       "(%s%s%s::%d)\n",
	       txdesc.retry_count, txdesc.tx_rate, fc,
	       WLAN_FC_GET_TYPE(fc) == WLAN_FC_TYPE_MGMT ? "Mgmt" : "",
	       WLAN_FC_GET_TYPE(fc) == WLAN_FC_TYPE_CTRL ? "Ctrl" : "",
	       WLAN_FC_GET_TYPE(fc) == WLAN_FC_TYPE_DATA ? "Data" : "",
	       WLAN_FC_GET_STYPE(fc));
	PDEBUG(DEBUG_EXTRA, "   addr1=" MACSTR " addr2=" MACSTR " addr3="
	       MACSTR "\n",
	       MAC2STR(txdesc.addr1), MAC2STR(txdesc.addr2),
	       MAC2STR(txdesc.addr3));

	if (local->iw_mode != IW_MODE_MASTER)
		return;

	spin_lock(&local->ap->sta_table_lock);
	/* FIX: is addr1 correct for all frame types? */
	sta = ap_get_sta(local->ap, txdesc.addr1);
	if (sta) {
		sta->tx_since_last_failure = 0;
		if (sta->tx_rate_idx > 0 &&
		    txdesc.tx_rate <= sta->tx_rate) {
			int rate = sta->tx_rate_idx;
			/* use next lower rate */
			while (rate > 0) {
				rate--;
				if (sta->tx_supp_rates & (1 << rate)) {
					sta->tx_rate_idx = rate;
					break;
				}
			}
			switch (sta->tx_rate_idx) {
			case 0: sta->tx_rate = 10; break;
			case 1: sta->tx_rate = 20; break;
			case 2: sta->tx_rate = 55; break;
			case 3: sta->tx_rate = 110; break;
			default: sta->tx_rate = 0; break;
			}
			PDEBUG(DEBUG_AP, "%s: STA " MACSTR " TX rate lowered "
			       "to %d\n", dev->name, MAC2STR(sta->addr),
			       sta->tx_rate);
		}
	} else {
		PDEBUG(DEBUG_AP, "Could not find STA for this TX error\n");
	}
	spin_unlock(&local->ap->sta_table_lock);
}


static const char* hfa384x_linkstatus_str(u16 linkstatus)
{
	switch (linkstatus) {
	case HFA384X_LINKSTATUS_CONNECTED:
		return "Connected";
	case HFA384X_LINKSTATUS_DISCONNECTED:
		return "Disconnected";
	case HFA384X_LINKSTATUS_AP_CHANGE:
		return "Access point change";
	case HFA384X_LINKSTATUS_AP_OUT_OF_RANGE:
		return "Access point out of range";
	case HFA384X_LINKSTATUS_AP_IN_RANGE:
		return "Access point in range";
	case HFA384X_LINKSTATUS_ASSOC_FAILED:
		return "Association failed";
	default:
		return "Unknown";
	}
}


/* Called only from hardware IRQ */
static void prism2_info(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 fid, val, *pos, len;
	int res, i, left;
	struct hfa384x_info_frame info;
	unsigned char buf[1024];
	struct hfa384x_comm_tallies *tallies;
	struct hfa384x_scan_result *scanres;

	fid = HFA384X_INW(HFA384X_INFOFID_OFF);

	res = hfa384x_setup_bap(dev, BAP1, fid, 0);
	if (!res)
		res = hfa384x_from_bap(dev, BAP1, &info, sizeof(info));
	if (res) {
		printk("Could not get info frame\n");
		if (res == -ETIMEDOUT)
			prism2_hw_reset(dev);
		return;
	}

	if (__le16_to_cpu(info.len) & 0x8000) {
		/* data register seems to give 0x8000 in some error cases even
		 * though busy bit is not set in offset register; re-reading
		 * the data seems to fix at least some of the cases */
		printk(KERN_DEBUG "%s: prism2_info: re-reading header to fix "
		       "possibly corrupted BAP read\n", dev->name);
		res = hfa384x_setup_bap(dev, BAP1, fid, 0);
		if (!res)
			res = hfa384x_from_bap(dev, BAP1, &info, sizeof(info));
		if (res)
			printk(KERN_DEBUG "prism2_info: could not re-read "
			       "info header\n");
		if (__le16_to_cpu(info.len) & 0x8000) {
			printk(KERN_DEBUG "prism2_info: re-read did not fix "
			       "header - ignoring this frame\n");
			return;
		}
	}

	info.len = __le16_to_cpu(info.len);
	info.type = __le16_to_cpu(info.type);

	if (info.type != HFA384X_INFO_COMMTALLIES)
		PDEBUG(DEBUG_EXTRA, "%s: INFO - fid=0x%04x - len=%d "
		       "type=0x%04x\n", dev->name, fid, info.len, info.type);

	switch (info.type) {
	case HFA384X_INFO_COMMTALLIES:
		if (info.len != 22 ||
		    hfa384x_from_bap(dev, BAP1, &buf,
				     sizeof(struct hfa384x_comm_tallies))) {
			printk("  info communication tallies failed\n");
			break;
		}
		tallies = (struct hfa384x_comm_tallies *) buf;
#define ADD_COMM_TALLIES(name) local->comm_tallies.name += tallies->name
		ADD_COMM_TALLIES(tx_unicast_frames);
		ADD_COMM_TALLIES(tx_multicast_frames);
		ADD_COMM_TALLIES(tx_fragments);
		ADD_COMM_TALLIES(tx_unicast_octets);
		ADD_COMM_TALLIES(tx_multicast_octets);
		ADD_COMM_TALLIES(tx_deferred_transmissions);
		ADD_COMM_TALLIES(tx_single_retry_frames);
		ADD_COMM_TALLIES(tx_multiple_retry_frames);
		ADD_COMM_TALLIES(tx_retry_limit_exceeded);
		ADD_COMM_TALLIES(tx_discards);
		ADD_COMM_TALLIES(rx_unicast_frames);
		ADD_COMM_TALLIES(rx_multicast_frames);
		ADD_COMM_TALLIES(rx_fragments);
		ADD_COMM_TALLIES(rx_unicast_octets);
		ADD_COMM_TALLIES(rx_multicast_octets);
		ADD_COMM_TALLIES(rx_fcs_errors);
		ADD_COMM_TALLIES(rx_discards_no_buffer);
		ADD_COMM_TALLIES(tx_discards_wrong_sa);
		ADD_COMM_TALLIES(rx_discards_wep_undecryptable);
		ADD_COMM_TALLIES(rx_message_in_msg_fragments);
		ADD_COMM_TALLIES(rx_message_in_bad_msg_fragments);
		break;

	case HFA384X_INFO_LINKSTATUS:
		if (info.len != 2 || hfa384x_from_bap(dev, BAP1, &val, 2)) {
			printk("  info linkstatus failed\n");
			break;
		}
		val = __le16_to_cpu(val);
		PDEBUG(DEBUG_EXTRA, "  LinkStatus=%d (%s)\n",
		       val, hfa384x_linkstatus_str(val));
		break;

	case HFA384X_INFO_SCANRESULTS:
		PDEBUG(DEBUG_EXTRA, "  ScanResults:\n");
		if (info.len < 3) {
			printk("    info scanresult failed\n");
			break;
		}
		val = (info.len - 1) * 2;
		if (val > sizeof(buf))
			val = sizeof(buf);
		if (hfa384x_from_bap(dev, BAP1, &buf, val)) {
			printk("    info read failed\n");
			break;
		}

		pos = (u16 *) buf;
		PDEBUG(DEBUG_EXTRA, "    Reserved=0x%04x\n",
		       __le16_to_cpu(*pos));
		pos++;
		PDEBUG(DEBUG_EXTRA, "    ScanReason=%d\n",
		       __le16_to_cpu(*pos));
		pos++;

		left = (info.len - 3) * 2;
		scanres = (struct hfa384x_scan_result *) pos;
		val = 1;
		while (left >= sizeof(struct hfa384x_scan_result)) {
			/* FIX: it might not be that wise to print a lot
			 * from here.. this is called from hw interrupt
			 * handler.. */
			PDEBUG(DEBUG_EXTRA, "    ScanResult %d:\n", val);
			PDEBUG(DEBUG_EXTRA, "      CHID=%d\n",
			       __le16_to_cpu(scanres->chid));
			PDEBUG(DEBUG_EXTRA, "      ANL=%d\n",
			       __le16_to_cpu(scanres->anl));
			PDEBUG(DEBUG_EXTRA, "      SL=%d\n",
			       __le16_to_cpu(scanres->sl));
			PDEBUG(DEBUG_EXTRA, "      bssid=" MACSTR "\n",
			       MAC2STR(scanres->bssid));
			PDEBUG(DEBUG_EXTRA, "      BcnInt=%d\n",
			       __le16_to_cpu(scanres->beacon_interval));
			PDEBUG(DEBUG_EXTRA, "      Capability=0x%04x\n",
			       __le16_to_cpu(scanres->capability));
			PDEBUG(DEBUG_EXTRA, "      SSID='");
			len = __le16_to_cpu(*(u16 *) &scanres->ssid);
			if (len > 32)
				len = 32;
			for (i = 0; i < len; i++) {
				unsigned char c = scanres->ssid[2 + i];
				if (c >= 32 && c < 127)
					PDEBUG2(DEBUG_EXTRA, "%c", c);
				else
					PDEBUG2(DEBUG_EXTRA, "<%02x>", c);
			}
			PDEBUG2(DEBUG_EXTRA, "'\n");
			PDEBUG(DEBUG_EXTRA, "      SupRates=");
			for (i = 0; i < sizeof(scanres->sup_rates); i++)
				PDEBUG2(DEBUG_EXTRA, "%02x ",
					scanres->sup_rates[i]);
			PDEBUG2(DEBUG_EXTRA, "\n");
			PDEBUG(DEBUG_EXTRA, "      Rate=%d\n",
			       __le16_to_cpu(scanres->rate));
			val++;
			left -= sizeof(struct hfa384x_scan_result);
			scanres++;
		}

		if (left > 0) {
			unsigned char *c = (unsigned char *) scanres;
			PDEBUG(DEBUG_EXTRA, "ScanRes extra: left=%d "
			       "(expected %dx):", left,
			       sizeof(struct hfa384x_scan_result));
			for (i = 0; i < left; i++)
				PDEBUG2(DEBUG_EXTRA, " %02x", *c++);
			PDEBUG2(DEBUG_EXTRA, "\n");
		}

		break;

	default:
		val = info.len > 0 ? (info.len - 1) * 2 : 0;
		if (val > sizeof(buf))
			val = sizeof(buf);
		if (val > 0 && hfa384x_from_bap(dev, BAP1, &buf, val)) {
			printk("  info read failed\n");
			break;
		}
		PDEBUG(DEBUG_EXTRA, "Unknown info frame:");
		for (i = 0; i < val; i++)
			PDEBUG2(DEBUG_EXTRA, " %02x", buf[i]);
		PDEBUG2(DEBUG_EXTRA, "\n");
	}
}


/* Called only from hardware IRQ */
static void prism2_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct net_device *dev = (struct net_device *) dev_id;
	local_info_t *local = (local_info_t *) dev->priv;
	int events = 0;
	u16 ev;
#ifndef final_version
	static long int last_magic_err = 0;
#endif

	if ((local->link->state & (DEV_PRESENT | DEV_CONFIG)) !=
	    (DEV_PRESENT | DEV_CONFIG)) {
		printk("%s: Interrupt, but dev not OK\n", dev->name);
		return;
	}

#ifndef final_version
	if (HFA384X_INW(HFA384X_SWSUPPORT0_OFF) != HFA384X_MAGIC) {
		HFA384X_OUTW(0xffff, HFA384X_EVACK_OFF);
		if (jiffies - last_magic_err > 10 * HZ)
			printk("%s: Interrupt, but SWSUPPORT0 does not match"
			       "%04X != %04X - card removed?\n", dev->name,
			       HFA384X_INW(HFA384X_SWSUPPORT0_OFF),
			       HFA384X_MAGIC);
		last_magic_err = jiffies;
		prism2_hw_reset(dev);
		return;
	}
#endif

	if (!local->hw_ready) {
		ev = HFA384X_INW(HFA384X_EVSTAT_OFF);
		printk(KERN_DEBUG "%s: prism2_interrupt: hw not ready; "
		       "skipping events 0x%04x\n", dev->name, ev);
		HFA384X_OUTW(ev, HFA384X_EVACK_OFF);
	}

	do {
		ev = HFA384X_INW(HFA384X_EVSTAT_OFF);
		if ((ev & HFA384X_EVENT_MASK) == 0)
			break;

		if (ev & HFA384X_EV_TX) {
			local->stats.tx_packets++;
			PDEBUG(DEBUG_FID, "interrupt: TX - fid=0x%04x\n",
			       HFA384X_INW(HFA384X_TXCOMPLFID_OFF));
			HFA384X_OUTW(HFA384X_EV_TX, HFA384X_EVACK_OFF);
		}

		if (ev & HFA384X_EV_TXEXC) {
			prism2_txexc(dev);
			HFA384X_OUTW(HFA384X_EV_TXEXC, HFA384X_EVACK_OFF);
		}

		if (ev & HFA384X_EV_RX) {
			prism2_rx(dev);
			HFA384X_OUTW(HFA384X_EV_RX, HFA384X_EVACK_OFF);
		}

		if (ev & HFA384X_EV_ALLOC) {
			prism2_alloc_ev(dev);
			HFA384X_OUTW(HFA384X_EV_ALLOC, HFA384X_EVACK_OFF);
		}

		if (ev & HFA384X_EV_WTERR) {
			PDEBUG(DEBUG_EXTRA, "%s: WTERR event\n", dev->name);
			HFA384X_OUTW(HFA384X_EV_WTERR, HFA384X_EVACK_OFF);
		}

		if (ev & HFA384X_EV_DTIM) {
			PDEBUG(DEBUG_EXTRA, "%s: DTIM event\n", dev->name);
			HFA384X_OUTW(HFA384X_EV_DTIM, HFA384X_EVACK_OFF);
		}

		if (ev & HFA384X_EV_TICK) {
			HFA384X_OUTW(HFA384X_EV_TICK, HFA384X_EVACK_OFF);
		}

		if (ev & HFA384X_EV_INFO) {
			prism2_info(dev);
			HFA384X_OUTW(HFA384X_EV_INFO, HFA384X_EVACK_OFF);
		}

		events++;
	} while (events < 20);

	if (events >= 20)
		PDEBUG(DEBUG_EXTRA, "prism2_interrupt: >20 events\n");
}


#define CS_CHECK(fn, args...) \
while ((last_ret = CardServices(last_fn = (fn), args)) != 0) goto cs_failed

#define CFG_CHECK(fn, args...) \
if (CardServices(fn, args) != 0) goto next_entry

#define CFG_CHECK2(fn, args...) \
do { int ret = CardServices(fn, args); \
if (ret != 0) { \
	PDEBUG(DEBUG_EXTRA, "CardServices(" #fn ") returned %d\n", ret); \
	cs_error(link->handle, fn, ret); \
	goto next_entry; \
} \
} while (0)


/* run after a CARD_INSERTATION event is received to configure the PCMCIA
 * socket and make the device available to the system */
static int prism2_config(dev_link_t *link)
{
	struct net_device *dev = (struct net_device *) link->priv;
	local_info_t *local = (local_info_t *) dev->priv;


	tuple_t tuple;
	cisparse_t parse;
	int last_fn, last_ret;
	u_char buf[64];
	config_info_t conf;
	win_req_t req;
	memreq_t map;
	cistpl_cftable_entry_t dflt = { 0 };

	PDEBUG(DEBUG_FLOW, "prism2_config()\n");

	tuple.DesiredTuple = CISTPL_CONFIG;
	tuple.Attributes = 0;
	tuple.TupleData = buf;
	tuple.TupleDataMax = sizeof(buf);
	tuple.TupleOffset = 0;
	CS_CHECK(GetFirstTuple, link->handle, &tuple);
	CS_CHECK(GetTupleData, link->handle, &tuple);
	CS_CHECK(ParseTuple, link->handle, &tuple, &parse);
	link->conf.ConfigBase = parse.config.base;
	link->conf.Present = parse.config.rmask[0];

	CS_CHECK(GetConfigurationInfo, link->handle, &conf);
	PDEBUG(DEBUG_HW, "%s: setting Vcc=%d (from config)\n", dev_info,
	       conf.Vcc);
	link->conf.Vcc = conf.Vcc;

	tuple.DesiredTuple = CISTPL_MANFID;
	tuple.Attributes = TUPLE_RETURN_COMMON;
	tuple.Attributes = 0;
	tuple.TupleData = buf;
	tuple.TupleDataMax = sizeof(buf);

	if (CardServices(GetFirstTuple, link->handle, &tuple) == CS_SUCCESS &&
	    CardServices(GetTupleData, link->handle, &tuple) == CS_SUCCESS &&
	    CardServices(ParseTuple, link->handle, &tuple, &parse) ==
	    CS_SUCCESS) {
		local->manfid = parse.manfid.manf;
		local->prodid = parse.manfid.card;
		PDEBUG(DEBUG_EXTRA, "CISTPL_MANFID: 0x%04x, 0x%04x\n",
		       local->manfid, local->prodid);

		/* 0x014d,0x0001 Symbol LA4111 Spectrum24
		 * 0x0101,0x0001 3Com 3CRWE737A AirConnect
		 * 0x0089,0x0001 Intel PRO/Wireless 2011 */
		if ((local->manfid == 0x014d && local->prodid == 0x0001) ||
		    (local->manfid == 0x0101 && local->prodid == 0x0001) ||
		    (local->manfid == 0x0089 && local->prodid == 0x0001)) {
			PDEBUG(DEBUG_EXTRA, "Symbol-based card\n");
			local->is_symbol = 1;
		}

		if (local->manfid == 0x0156 && local->prodid == 0x0002) {
			PDEBUG(DEBUG_EXTRA, "Lucent-based card\n");
			local->is_lucent = 1;
		}
	}

	/* Look for an appropriate configuration table entry in the CIS */
	tuple.DesiredTuple = CISTPL_CFTABLE_ENTRY;
	CS_CHECK(GetFirstTuple, link->handle, &tuple);
	for (;;) {
		cistpl_cftable_entry_t *cfg = &(parse.cftable_entry);
		CFG_CHECK2(GetTupleData, link->handle, &tuple);
		CFG_CHECK2(ParseTuple, link->handle, &tuple, &parse);

		if (cfg->flags & CISTPL_CFTABLE_DEFAULT)
			dflt = *cfg;
		if (cfg->index == 0)
			goto next_entry;
		link->conf.ConfigIndex = cfg->index;
		PDEBUG(DEBUG_EXTRA, "Checking CFTABLE_ENTRY 0x%02X "
		       "(default 0x%02X)\n", cfg->index, dflt.index);
	
		/* Does this card need audio output? */
		if (cfg->flags & CISTPL_CFTABLE_AUDIO) {
			link->conf.Attributes |= CONF_ENABLE_SPKR;
			link->conf.Status = CCSR_AUDIO_ENA;
		}
	
		/* Use power settings for Vcc and Vpp if present */
		/*  Note that the CIS values need to be rescaled */
		if (cfg->vcc.present & (1 << CISTPL_POWER_VNOM)) {
			if (conf.Vcc != cfg->vcc.param[CISTPL_POWER_VNOM] /
			    10000)
				goto next_entry;
		} else if (dflt.vcc.present & (1 << CISTPL_POWER_VNOM)) {
			if (conf.Vcc != dflt.vcc.param[CISTPL_POWER_VNOM] /
			    10000)
				goto next_entry;
		}
	    
		if (cfg->vpp1.present & (1 << CISTPL_POWER_VNOM))
			link->conf.Vpp1 = link->conf.Vpp2 =
				cfg->vpp1.param[CISTPL_POWER_VNOM] / 10000;
		else if (dflt.vpp1.present & (1 << CISTPL_POWER_VNOM))
			link->conf.Vpp1 = link->conf.Vpp2 =
				dflt.vpp1.param[CISTPL_POWER_VNOM] / 10000;
	
		/* Do we need to allocate an interrupt? */
		if (cfg->irq.IRQInfo1 || dflt.irq.IRQInfo1)
			link->conf.Attributes |= CONF_ENABLE_IRQ;
		else if (!(link->conf.Attributes & CONF_ENABLE_IRQ)) {
			/* At least Compaq WL200 does not have IRQInfo1 set,
			 * but it does not work without interrupts.. */
			printk("Config has no IRQ info, but trying to enable "
			       "IRQ anyway..\n");
			link->conf.Attributes |= CONF_ENABLE_IRQ;
		}
	
		/* IO window settings */
		PDEBUG(DEBUG_EXTRA, "IO window settings: cfg->io.nwin=%d "
		       "dflt.io.nwin=%d\n",
		       cfg->io.nwin, dflt.io.nwin);
		link->io.NumPorts1 = link->io.NumPorts2 = 0;
		if ((cfg->io.nwin > 0) || (dflt.io.nwin > 0)) {
			cistpl_io_t *io = (cfg->io.nwin) ? &cfg->io : &dflt.io;
			link->io.Attributes1 = IO_DATA_PATH_WIDTH_AUTO;
			PDEBUG(DEBUG_EXTRA, "io->flags = 0x%04X, "
			       "io.base=0x%04x, len=%d\n", io->flags,
			       io->win[0].base, io->win[0].len);
			if (!(io->flags & CISTPL_IO_8BIT))
				link->io.Attributes1 = IO_DATA_PATH_WIDTH_16;
			if (!(io->flags & CISTPL_IO_16BIT))
				link->io.Attributes1 = IO_DATA_PATH_WIDTH_8;
			link->io.IOAddrLines = io->flags &
				CISTPL_IO_LINES_MASK;
			link->io.BasePort1 = io->win[0].base;
			link->io.NumPorts1 = io->win[0].len;
			if (io->nwin > 1) {
				link->io.Attributes2 = link->io.Attributes1;
				link->io.BasePort2 = io->win[1].base;
				link->io.NumPorts2 = io->win[1].len;
			}
		}

		/* This reserves IO space but doesn't actually enable it */
		CFG_CHECK2(RequestIO, link->handle, &link->io);

		/* Now set up a common memory window, if needed.  There is room
		 * in the dev_link_t structure for one memory window handle,
		 * but if the base addresses need to be saved, or if multiple
		 * windows are needed, the info should go in the private data
		 * structure for this device.
		 *
		 * Note that the memory window base is a physical address, and
		 * needs to be mapped to virtual space with ioremap() before it
		 * is used.
		 */
		if ((cfg->mem.nwin > 0) || (dflt.mem.nwin > 0)) {
			cistpl_mem_t *mem =
				(cfg->mem.nwin) ? &cfg->mem : &dflt.mem;
			req.Attributes = WIN_DATA_WIDTH_16 |
				WIN_MEMORY_TYPE_CM | WIN_ENABLE;
			req.Base = mem->win[0].host_addr;
			req.Size = mem->win[0].len;

			if (local->is_symbol) {
				/* Symbol-based cards seems to require
				 * following settings: */
				printk("Symbol-based card - using different "
				       "common memory window settings\n");
				req.Attributes = WIN_DATA_WIDTH_8 |
					WIN_MEMORY_TYPE_CM |
					WIN_ADDR_SPACE_MEM |
					WIN_ENABLE;
				req.Size = 0;
			}

			req.AccessSpeed = 0;
			PDEBUG(DEBUG_EXTRA, "Requesting common memory window: "
			       "Base=0x%04X, Size=0x%04X\n",
			       (unsigned int) req.Base,
			       req.Size);
			link->win = (window_handle_t)link->handle;
			CFG_CHECK2(RequestWindow, &link->win, &req);
			map.Page = 0; map.CardOffset = mem->win[0].card_addr;
			CFG_CHECK2(MapMemPage, link->win, &map);
		}
		/* This configuration table entry is OK */
		break;

	next_entry:
		CS_CHECK(GetNextTuple, link->handle, &tuple);
	}

	/*
	 * Allocate an interrupt line.  Note that this does not assign a
	 * handler to the interrupt, unless the 'Handler' member of the
	 * irq structure is initialized.
	 */
	if (link->conf.Attributes & CONF_ENABLE_IRQ) {
		int i;
		link->irq.Attributes = IRQ_TYPE_EXCLUSIVE | IRQ_HANDLE_PRESENT;
		link->irq.IRQInfo1 = IRQ_INFO2_VALID | IRQ_LEVEL_ID;
		if (irq_list[0] == -1)
			link->irq.IRQInfo2 = irq_mask;
		else
			for (i = 0; i < 4; i++)
				link->irq.IRQInfo2 |= 1 << irq_list[i];
		link->irq.Handler = (void *) prism2_interrupt;
		link->irq.Instance = dev;
		CS_CHECK(RequestIRQ, link->handle, &link->irq);
	}
	
	/*
	 * This actually configures the PCMCIA socket -- setting up
	 * the I/O windows and the interrupt mapping, and putting the
	 * card and host interface into "Memory and IO" mode.
	 */
	CS_CHECK(RequestConfiguration, link->handle, &link->conf);

	dev->irq = link->irq.AssignedIRQ;
	dev->base_addr = link->io.BasePort1;

	/* Finally, report what we've done */
	printk(KERN_INFO "%s: index 0x%02x: Vcc %d.%d",
	       dev_info, link->conf.ConfigIndex,
	       link->conf.Vcc / 10, link->conf.Vcc % 10);
	if (link->conf.Vpp1)
		printk(", Vpp %d.%d", link->conf.Vpp1 / 10,
		       link->conf.Vpp1 % 10);
	if (link->conf.Attributes & CONF_ENABLE_IRQ)
		printk(", irq %d", link->irq.AssignedIRQ);
	if (link->io.NumPorts1)
		printk(", io 0x%04x-0x%04x", link->io.BasePort1,
		       link->io.BasePort1+link->io.NumPorts1-1);
	if (link->io.NumPorts2)
		printk(" & 0x%04x-0x%04x", link->io.BasePort2,
		       link->io.BasePort2+link->io.NumPorts2-1);
	if (link->win)
		printk(", mem 0x%06lx-0x%06lx", req.Base,
		       req.Base+req.Size-1);
	printk("\n");

	link->state |= DEV_CONFIG;
	link->state &= ~DEV_CONFIG_PENDING;


#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
	{
		int i = 0;
		do {
			sprintf(dev->name, "wlan%d", i++);
		} while (dev_get(dev->name));
	}
#else
	memcpy(dev->name, "wlan%d", 7);
#endif
	if (register_netdev(dev)) {
		printk(KERN_WARNING "%s: register_netdev() failed!\n",
		       dev_info);
		prism2_release((u_long)link);
		return 1;
	}
	printk(KERN_INFO "%s: Registered netdevice %s\n", dev_info, dev->name);

	prism2_init_proc(local);
	ap_init_data(local);

	copy_dev_name(local->node, dev);
	link->dev = &local->node;

	return prism2_hw_config(dev, 1);

 cs_failed:
	cs_error(link->handle, last_fn, last_ret);
	prism2_release((u_long)link);
	return 1;
}


static void prism2_release(u_long arg)
{
	dev_link_t *link = (dev_link_t *)arg;
	struct net_device *dev = (struct net_device *) link->priv;

	PDEBUG(DEBUG_FLOW, "prism2_release\n");

	if (link->open) {
		printk("%s: release postponed, '%s' still open\n",
		      dev_info, link->dev->dev_name);
		if (link->state & DEV_PRESENT) {
			link->state |= DEV_STALE_CONFIG;
			return;
		} else {
			/* FIX - this should not happen.. prism2_close()
			 * should be called from somewhere else */
			printk("Card already removed - closing device\n");
			prism2_close(dev);
		}
	}

	PDEBUG(DEBUG_FLOW, "release -> prism2_hw_shutdown\n");
	if (dev != NULL)
		prism2_hw_shutdown(dev);
	PDEBUG(DEBUG_FLOW, "release <- prism2_hw_shutdown\n");


	if (link->win)
		CardServices(ReleaseWindow, link->win);
	CardServices(ReleaseConfiguration, link->handle);
	if (link->io.NumPorts1)
		CardServices(ReleaseIO, link->handle, &link->io);
	if (link->irq.AssignedIRQ)
		CardServices(ReleaseIRQ, link->handle, &link->irq);

	PDEBUG(DEBUG_FLOW, "release: CardServices(Release*) done\n");


	link->state &= ~DEV_CONFIG;

	if (link->state & DEV_STALE_LINK) {
		/* FIX - should this be here? */
		PDEBUG(DEBUG_FLOW, "prism2_release => prism2_detach\n");
		prism2_detach(link);
	}

	PDEBUG(DEBUG_FLOW, "release - done\n");
}



static int prism2_event(event_t event, int priority,
			event_callback_args_t *args)
{
	dev_link_t *link = args->client_data;
	struct net_device *dev = (struct net_device *) link->priv;

	switch (event) {
	case CS_EVENT_CARD_INSERTION:
		PDEBUG(DEBUG_EXTRA, "%s: CS_EVENT_CARD_INSERTION\n", dev_info);
		link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
		if (prism2_config(link))
			dev->irq = 0;
		break;

	case CS_EVENT_CARD_REMOVAL:
		PDEBUG(DEBUG_EXTRA, "%s: CS_EVENT_CARD_REMOVAL\n", dev_info);
		link->state &= ~DEV_PRESENT;
		if (link->state & DEV_CONFIG) {
			netif_stop_queue(dev);
			netif_device_detach(dev);
			mod_timer(&link->release, jiffies + HZ / 20);
		}
		break;

	case CS_EVENT_PM_SUSPEND:
		PDEBUG(DEBUG_EXTRA, "%s: CS_EVENT_PM_SUSPEND\n", dev_info);
		link->state |= DEV_SUSPEND;
		/* fall through */

	case CS_EVENT_RESET_PHYSICAL:
		PDEBUG(DEBUG_EXTRA, "%s: CS_EVENT_RESET_PHYSICAL\n", dev_info);
		if (link->state & DEV_CONFIG) {
			if (link->open) {
				netif_stop_queue(dev);
				netif_device_detach(dev);
			}
			CardServices(ReleaseConfiguration, link->handle);
		}
		break;

	case CS_EVENT_PM_RESUME:
		PDEBUG(DEBUG_EXTRA, "%s: CS_EVENT_PM_RESUME\n", dev_info);
		link->state &= ~DEV_SUSPEND;
		/* fall through */

	case CS_EVENT_CARD_RESET:
		PDEBUG(DEBUG_EXTRA, "%s: CS_EVENT_CARD_RESET\n", dev_info);
		if (link->state & DEV_CONFIG) {
			CardServices(RequestConfiguration, link->handle,
				     &link->conf);
			if (link->open) {
				prism2_hw_shutdown(dev);
				prism2_hw_config(dev, 0);
				netif_device_attach(dev);
				netif_start_queue(dev);
			}
		}
		break;

	default:
		PDEBUG(DEBUG_EXTRA, "%s: prism2_event() - unknown event %d\n",
		       dev_info, event);
		break;
	}
	return 0;
}


static int __init init_prism2(void)
{
	servinfo_t serv;

	printk(KERN_INFO "%s: %s\n"
	       "%s: (c) SSH Communications Security Corp <jkm@ssh.com>\n",
	       dev_info, version, dev_info);
	CardServices(GetCardServicesInfo, &serv);
	if (serv.Revision != CS_RELEASE_CODE) {
		printk(KERN_NOTICE
		       "%s: CardServices release does not match!\n", dev_info);
		return -1;
	}
	register_pccard_driver(&dev_info, &prism2_attach, &prism2_detach);

	return 0;
}


static void __exit exit_prism2(void)
{
	unregister_pccard_driver(&dev_info);
	while (dev_list) {
		PDEBUG(DEBUG_FLOW, "exit_prism2 - detaching device\n");
		del_timer(&dev_list->release);
		if (dev_list->state & DEV_CONFIG)
			prism2_release((u_long)dev_list);
		prism2_detach(dev_list);
	}

	if (prism2_proc != NULL)
		remove_proc_entry("prism2", proc_net);

	printk(KERN_INFO "%s: Driver unloaded\n", dev_info);
}


module_init(init_prism2);
module_exit(exit_prism2);
