/*
 * Intersil Prism2 driver with Host AP (software access point) support
 * Copyright 2001, SSH Communications Security Corp
 * Jouni Malinen <jkm@ssh.com>
 *
 * This file is to be included into prism2.c when S/W AP functionality is
 * compiled.
 *
 * AP:  FIX:
 * - if unicast Class 2 (assoc,reassoc,disassoc) frame received from
 *   unauthenticated STA, send deauth. frame (8802.11: 5.5)
 * - if unicast Class 3 (data with to/from DS,deauth,pspoll) frame received
 *   from authenticated, but unassoc STA, send disassoc frame (8802.11: 5.5)
 * - if unicast Class 3 received from unauthenticated STA, send deauth. frame
 *   (8802.11: 5.5)
 */


static int ap_debug_proc_read(char *page, char **start, off_t off,
			      int count, int *eof, void *data)
{
	char *p = page;
	struct ap_data *ap = (struct ap_data *) data;
	int i;
	unsigned long flags;

	if (off != 0) {
		*eof = 1;
		return 0;
	}

	p += sprintf(p, "BridgedUnicastFrames=%u\n", ap->bridged_unicast);
	p += sprintf(p, "BridgedMulticastFrames=%u\n", ap->bridged_multicast);

	spin_lock_irqsave(&ap->ap_queue_lock, flags);
	for (i = 0; i < AP_QUEUE_LEN; i++) {
		if (ap->ap_queued_items[i] != NULL)
			p += sprintf(p, "AP_QUEUE[%d]\n", i);
	}
	spin_unlock_irqrestore(&ap->ap_queue_lock, flags);

	return (p - page);
}


static void ap_sta_hash_add(struct ap_data *ap, struct sta_info *sta)
{
	sta->hnext = ap->sta_hash[STA_HASH(sta->addr)];
	ap->sta_hash[STA_HASH(sta->addr)] = sta;
}

static void ap_sta_hash_del(struct ap_data *ap, struct sta_info *sta)
{
	struct sta_info *s;

	s = ap->sta_hash[STA_HASH(sta->addr)];
	if (s == NULL) return;
	if (memcmp(s->addr, sta->addr, 6) == 0) {
		ap->sta_hash[STA_HASH(sta->addr)] = s->hnext;
		return;
	}

	while (s->hnext != NULL && memcmp(s->hnext->addr, sta->addr, 6) != 0)
		s = s->hnext;
	if (s->hnext != NULL)
		s->hnext = s->hnext->hnext;
	else
		printk("AP: could not remove STA " MACSTR " from hash table\n",
		       MAC2STR(sta->addr));
}

static void ap_free_sta(struct ap_data *ap, struct sta_info *sta)
{
	if (ap->proc != NULL) {
		char name[20];
		sprintf(name, MACSTR, MAC2STR(sta->addr));
		remove_proc_entry(name, ap->proc);
	}

	ap_sta_hash_del(ap, sta);
	if (sta->aid > 0)
		ap->sta_aid[sta->aid - 1] = NULL;

	if (sta->tx_buf_head != NULL) {
		struct sta_buffer_frame *prev, *fbuf =
			sta->tx_buf_head;
		while (fbuf != NULL) {
			prev = fbuf;
			dev_kfree_skb(fbuf->skb);
			fbuf = fbuf->next;
			prev->next = NULL;
			kfree(prev);
			sta->buffer_count--;
		}
		sta->tx_buf_head = sta->tx_buf_tail = NULL;
	}
	list_del(&sta->list);
	kfree(sta);
}

static void ap_handle_timer(unsigned long data)
{
	local_info_t *local = (local_info_t *) data;
	struct ap_data *ap;
	struct list_head *ptr;
	unsigned long flags;

	if (local == NULL || local->ap == NULL)
		return;

	ap = local->ap;

	spin_lock_irqsave(&ap->sta_table_lock, flags);

	for (ptr = ap->sta_list.next; ptr != NULL && ptr != &ap->sta_list;
	     ptr = ptr->next) {
		unsigned long last;
		struct sta_info *sta = (struct sta_info *) ptr;
		if (atomic_read(&sta->users) != 0) {
			PDEBUG(DEBUG_AP, "ap_handle_timer: STA " MACSTR
			       " users=%i\n",
			       MAC2STR(sta->addr), atomic_read(&sta->users));
			continue;
		}
		last = sta->last_rxtx;
		if (sta->last_assoc > last)
			last = sta->last_assoc;
		if (sta->last_auth > last)
			last = sta->last_auth;
		if (last > jiffies || last + AP_MAX_INACTIVITY < jiffies) {
			PDEBUG(DEBUG_AP, "STA " MACSTR " inactivity timeout "
			       "(last=%ld, jiffies=%ld)\n",
			       MAC2STR(sta->addr), last, jiffies);
			if (sta->aid > 0 && sta->tx_buf_head != NULL)
				prism2_set_tim(local->dev, sta->aid, 0);
			ap_free_sta(ap, sta);
		}
	}
	spin_unlock_irqrestore(&ap->sta_table_lock, flags);

	ap->timer.expires = jiffies + AP_CHECK_INTERVAL;
	add_timer(&ap->timer);
}

void ap_init_data(local_info_t *local)
{
	struct ap_data *ap = local->ap;

	if (ap == NULL) {
		printk(KERN_WARNING "ap_init_data: ap == NULL\n");
		return;
	}
	memset(ap, 0, sizeof(struct ap_data));

	ap->proc = local->proc;
	ap->bridge_packets = 1;

	spin_lock_init(&ap->ap_queue_lock);
	spin_lock_init(&ap->sta_table_lock);
	INIT_LIST_HEAD(&ap->sta_list);

	init_timer(&ap->timer);
	ap->timer.expires = jiffies + AP_CHECK_INTERVAL;
	ap->timer.data = (unsigned long) local;
	ap->timer.function = ap_handle_timer;
	add_timer(&ap->timer);

	if (ap->proc != NULL) {
		create_proc_read_entry("ap_debug", 0, ap->proc,
				       ap_debug_proc_read, ap);
	}

	/* Initialize task queue structure for AP management */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
	local->ap->ap_queue.next = NULL;
#else
	INIT_LIST_HEAD(&local->ap->ap_queue.list);
#endif
	local->ap->ap_queue.sync = 0;
	local->ap->ap_queue.routine = (void (*)(void *))handle_ap_queue;
	local->ap->ap_queue.data = local;
}

void ap_free_data(struct ap_data *ap)
{
	struct list_head *ptr;
	int i;

	if (ap == NULL) {
		printk(KERN_WARNING "ap_free_data: ap == NULL\n");
		return;
	}

	del_timer(&ap->timer);

	for (i = 0; i < AP_QUEUE_LEN; i++) {
		if (ap->ap_queued_items[i] != NULL) {
			kfree(ap->ap_queued_items[i]);
			ap->ap_queued_items[i] = NULL;
		}
	}

	ptr = ap->sta_list.next;
	while (ptr != NULL && ptr != &ap->sta_list) {
		struct sta_info *sta = (struct sta_info *) ptr;
		ptr = ptr->next;
		ap_free_sta(ap, sta);
	}

	if (ap->proc != NULL) {
		remove_proc_entry("ap_debug", ap->proc);
	}
}


/* caller should have mutex for AP STA list handling */
struct sta_info* ap_get_sta(struct ap_data *ap, u8 *sta)
{
	struct sta_info *s;

	s = ap->sta_hash[STA_HASH(sta)];
	while (s != NULL && memcmp(s->addr, sta, 6) != 0)
		s = s->hnext;
	return s;
}

/* Called only from software IRQ */
static void prism2_send_mgmt(struct net_device *dev,
			     struct hfa384x_tx_frame *txdesc, char *body,
			     int body_len, int txevent)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int res, idx;

	txdesc->status = 0;
	txdesc->reserved1 = 0;
	txdesc->reserved2 = 0;
	txdesc->retry_count = 0;

	if (txevent)
		txdesc->tx_control = __cpu_to_le16(HFA384X_TX_CTRL_802_11 |
						   HFA384X_TX_CTRL_TX_EX |
						   HFA384X_TX_CTRL_TX_OK);
	else {
#ifdef PRISM2_NULLFUNC_ACK
		/* also HFA384X_TX_CTRL_ALT_RTRY could be set and retry
		 * count zero to prevent many ACK frames from being sent;
		 * unfortunately this does not, however, seem to be very good
		 * solution, probably because ACK timing is rather strict and
		 * retries seems to be the only way to get at least some ACK
		 * to match with a data::null_function.. */
#endif
		txdesc->tx_control = __cpu_to_le16(HFA384X_TX_CTRL_802_11);
	}

	txdesc->data_len = __cpu_to_le16(body_len);


	/* FIX: set tx_rate if f/w does not know how to do it */
	txdesc->tx_rate = 0;

	idx = prism2_get_txfid_idx(local);
	if (idx < 0)
		return;

	if (local->frame_dump & PRISM2_DUMP_TX_HDR)
		prism2_dump_tx_header(dev->name, txdesc);

	spin_lock_bh(&local->baplock);
	res = hfa384x_setup_bap(dev, BAP0, local->txfid[idx], 0);
	if (!res)
		res = hfa384x_to_bap(dev, BAP0, txdesc, sizeof(*txdesc));
	if (!res && body != NULL)
		res = hfa384x_to_bap(dev, BAP0, body, body_len);
	spin_unlock_bh(&local->baplock);
	if (!res)
		res = prism2_transmit(dev, idx);

	if (res) {
		local->intransmitfid[idx] = PRISM2_TXFID_EMPTY;
		printk("%s: prism2_send_mgmt - to BAP0 failed\n", dev->name);
		if (res == -ETIMEDOUT)
			prism2_hw_reset(dev);
	}
}


static int prism2_sta_proc_read(char *page, char **start, off_t off,
				int count, int *eof, void *data)
{
	char *p = page;
	struct sta_info *sta = (struct sta_info *) data;
	int i;

	/* FIX: possible race condition.. the STA data could have just expired,
	 * but proc entry was still here so that the read could have started;
	 * some locking should be done here.. */

	if (off != 0) {
		*eof = 1;
		return 0;
	}

	p += sprintf(p, "STA=" MACSTR "\nusers=%d\naid=%d\nflags=0x%04x\n"
		     "capability=0x%02x\nlisten_interval=%d\nsupported_rates=",
		     MAC2STR(sta->addr), atomic_read(&sta->users), sta->aid,
		     sta->flags, sta->capability, sta->listen_interval);
	for (i = 0; i < sizeof(sta->supported_rates); i++)
		if (sta->supported_rates[i] != 0)
			p += sprintf(p, "<%02x>", sta->supported_rates[i]);
	p += sprintf(p, "\njiffies=%ld\nlast_auth=%ld\nlast_assoc=%ld\n"
		     "last_rxtx=%ld\nrx_packets=%ld\ntx_packets=%ld\n"
		     "rx_bytes=%ld\ntx_bytes=%ld\nbuffer_count=%d\n"
		     "last_rx: silence=%d signal=%d rate=%d flow=%d\n"
		     "tx_rate=%d\ntx[1M]=%d\ntx[2M]=%d\ntx[5.5M]=%d\n"
		     "tx[11M]=%d\n"
		     "rx[1M]=%d\nrx[2M]=%d\nrx[5.5M]=%d\nrx[11M]=%d\n",
		     jiffies, sta->last_auth, sta->last_assoc, sta->last_rxtx,
		     sta->rx_packets, sta->tx_packets, sta->rx_bytes,
		     sta->tx_bytes, sta->buffer_count, sta->last_rx_silence,
		     sta->last_rx_signal, sta->last_rx_rate,
		     sta->last_rx_flow,
		     sta->tx_rate, sta->tx_count[0], sta->tx_count[1],
		     sta->tx_count[2], sta->tx_count[3],  sta->rx_count[0],
		     sta->rx_count[1], sta->rx_count[2], sta->rx_count[3]);

	return (p - page);
}


/* Called only from software IRQ */
static void handle_authen(local_info_t *local, struct hfa384x_rx_frame *rxdesc)
{
	struct net_device *dev = local->dev;
	char *body = (char *) (rxdesc + 1);
	int len;
	u16 auth_alg, auth_transaction, status_code, *pos;
	u16 resp = WLAN_STATUS_SUCCESS;
	struct hfa384x_tx_frame *txdesc;
	struct sta_info *sta = NULL;
	unsigned long flags;
	int new_sta = 0;

	len = __le16_to_cpu(rxdesc->data_len);

	if (len < 6) {
		printk("handle_authen - too short payload (len=%d)\n", len);
		return;
	}

	pos = (u16 *) body;
	auth_alg = __le16_to_cpu(*pos);
	pos++;
	auth_transaction = __le16_to_cpu(*pos);
	pos++;
	status_code = __le16_to_cpu(*pos);

	PDEBUG(DEBUG_AP, "%s: authentication: " MACSTR " len=%d, auth_alg=%d, "
	       "auth_transaction=%d, status_code=%d\n", dev->name,
	       MAC2STR(rxdesc->addr2), len, auth_alg, auth_transaction,
	       status_code);

	if (auth_alg != WLAN_AUTH_OPEN) {
		printk("Unknown authentication algorithm (%d)\n", auth_alg);
		resp = WLAN_STATUS_NOT_SUPPORTED_AUTH_ALG;
		goto fail;
	}

	if (auth_transaction != 1) {
		printk("Unknown authentication transaction number (%d)\n",
		       auth_transaction);
		resp = WLAN_STATUS_UNKNOWN_AUTH_TRANSACTION;
		goto fail;
	}

	spin_lock_irqsave(&local->ap->sta_table_lock, flags);
	sta = ap_get_sta(local->ap, rxdesc->addr2);
	if (sta == NULL) {
		PDEBUG(DEBUG_AP, "AP: new STA " MACSTR "\n",
		       MAC2STR(rxdesc->addr2));

		if (local->ap->num_sta >= MAX_STA_COUNT) {
			/* FIX: might try to remove some old STAs first? */
			spin_unlock_irqrestore(&local->ap->sta_table_lock,
					       flags);
			printk("AP: no more room for new STAs\n");
			resp = WLAN_STATUS_UNSPECIFIED_FAILURE;
			goto fail;
		}

		sta = (struct sta_info *)
			kmalloc(sizeof(struct sta_info), GFP_ATOMIC);
		if (sta == NULL) {
			spin_unlock_irqrestore(&local->ap->sta_table_lock,
					       flags);
			printk("AP: kmalloc failed\n");
			resp = WLAN_STATUS_UNSPECIFIED_FAILURE;
			goto fail;
		}
		memset(sta, 0, sizeof(struct sta_info));
		list_add(&sta->list, &local->ap->sta_list);
		local->ap->num_sta++;

		/* initialize STA info data */
		memcpy(&sta->addr, &rxdesc->addr2, 6);
		sta->aid = 0;
		sta->flags = WLAN_STA_AUTH;
		ap_sta_hash_add(local->ap, sta);

		new_sta = 1;
	}
	atomic_inc(&sta->users);
	spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);

	if (new_sta && local->ap->proc != NULL) {
		char name[20];
		sprintf(name, MACSTR, MAC2STR(sta->addr));
		sta->proc = create_proc_read_entry(name, 0,
						  local->ap->proc,
						  prism2_sta_proc_read, sta);

	}

 fail:
	/* use the queued buffer for transmission because it is large enough
	 * and not needed anymore */
	txdesc = (struct hfa384x_tx_frame *) rxdesc;

	txdesc->frame_control = __cpu_to_le16((WLAN_FC_TYPE_MGMT << 2) |
					      (WLAN_FC_STYPE_AUTH << 4));
	/* Addr1 = DA */
	memcpy(txdesc->addr1, txdesc->addr2, 6);
	/* Addr2 = SA */
	memcpy(txdesc->addr2, dev->dev_addr, 6);
	/* Addr3 = BSSID - already set */

	pos = (u16 *) body;
	*pos = __cpu_to_le16(WLAN_AUTH_OPEN); /* auth_alg */
	pos++;
	*pos = __cpu_to_le16(2); /* auth_transaction */
	pos++;
	*pos = __cpu_to_le16(resp); /* status_code */

	prism2_send_mgmt(dev, txdesc, body, 6, 1);

	if (sta) {
		sta->last_auth = jiffies;
		atomic_dec(&sta->users);
	}
}


static void prism2_check_tx_rates(struct sta_info *sta)
{
	int i;

	sta->tx_supp_rates = 0;
	for (i = 0; i < sizeof(sta->supported_rates); i++) {
		if ((sta->supported_rates[i] & 0x7f) == 2)
			sta->tx_supp_rates |= WLAN_RATE_1M;
		if ((sta->supported_rates[i] & 0x7f) == 4)
			sta->tx_supp_rates |= WLAN_RATE_2M;
		if ((sta->supported_rates[i] & 0x7f) == 11)
			sta->tx_supp_rates |= WLAN_RATE_5M5;
		if ((sta->supported_rates[i] & 0x7f) == 22)
			sta->tx_supp_rates |= WLAN_RATE_11M;
	}
	sta->tx_max_rate = sta->tx_rate = sta->tx_rate_idx = 0;
	if (sta->tx_supp_rates & WLAN_RATE_1M) {
		sta->tx_rate = 10;
		sta->tx_max_rate = sta->tx_rate_idx = 0;
	}
	if (sta->tx_supp_rates & WLAN_RATE_2M) {
		sta->tx_rate = 20;
		sta->tx_max_rate = sta->tx_rate_idx = 1;
	}
	if (sta->tx_supp_rates & WLAN_RATE_5M5) {
		sta->tx_rate = 55;
		sta->tx_max_rate = sta->tx_rate_idx = 2;
	}
	if (sta->tx_supp_rates & WLAN_RATE_11M) {
		sta->tx_rate = 110;
		sta->tx_max_rate = sta->tx_rate_idx = 3;
	}
}

/* Called only from software IRQ */
static void handle_assoc(local_info_t *local, struct hfa384x_rx_frame *rxdesc,
			 int reassoc)
{
	struct net_device *dev = local->dev;
	char body[12], *p;
	int len, left;
	u16 *pos;
	u16 resp = WLAN_STATUS_SUCCESS;
	struct hfa384x_tx_frame *txdesc;
	struct sta_info *sta = NULL;
	int send_deauth = 0;
	unsigned long flags;

	left = len = __le16_to_cpu(rxdesc->data_len);

	if (len < (reassoc ? 10 : 4)) {
		printk("handle_assoc - too short payload "
		       "(len=%d, reassoc=%d)\n", len, reassoc);
		return;
	}

	PDEBUG(DEBUG_AP, "%s: %sassociation request: " MACSTR " len=%d\n",
	       dev->name, reassoc ? "re" : "", MAC2STR(rxdesc->addr2), len);

	spin_lock_irqsave(&local->ap->sta_table_lock, flags);
	sta = ap_get_sta(local->ap, rxdesc->addr2);
	if (sta == NULL || (sta->flags & WLAN_STA_AUTH) == 0) {
		spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);
		printk("AP: STA " MACSTR " trying to associate before "
		       "authentication\n", MAC2STR(rxdesc->addr2));
		if (sta) {
			PDEBUG(DEBUG_AP, "sta: addr=" MACSTR
			       " aid=%d flags=0x%04x\n",
			       MAC2STR(sta->addr), sta->aid, sta->flags);
		}
		send_deauth = 1;
		resp = WLAN_STATUS_UNSPECIFIED_FAILURE;
		goto fail;
	}
	atomic_inc(&sta->users);
	spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);

	pos = (u16 *) (rxdesc + 1);
	sta->capability = __le16_to_cpu(*pos);
	pos++; left -= 2;
	sta->listen_interval = __le16_to_cpu(*pos);
	pos++; left -= 2;
	PDEBUG(DEBUG_AP, "  capability=0x%04x, listen_interval=%d",
	       sta->capability, sta->listen_interval);

	if (reassoc) {
		PDEBUG2(DEBUG_AP, ", current AP: " MACSTR,
			MAC2STR((u8 *) pos));
		pos++; pos++; pos++; left -= 6;
	}

	if (left >= 2) {
		unsigned int ileft;
		unsigned char *u = (unsigned char *) pos;

		if (*u == 0) {
			u++; left--;
			ileft = *u;
			u++; left--;

			if (ileft > left || ileft > MAX_SSID_LEN) {
				PDEBUG2(DEBUG_AP, " - SSID overflow\n");
				atomic_dec(&sta->users);
				return;
			}

			if (ileft != strlen(local->essid) ||
			    memcmp(local->essid, u, ileft) != 0) {
				PDEBUG2(DEBUG_AP, " - not our SSID\n");
				atomic_dec(&sta->users);
				return;
			}

			u += ileft;
			left -= ileft;
		}

		if (*u == 1) {
			u++; left--;
			ileft = *u;
			u++; left--;
			
			if (ileft > left || ileft == 0 || ileft > 8) {
				PDEBUG2(DEBUG_AP, " - SUPP_RATES len error\n");
				atomic_dec(&sta->users);
				return;
			}

			memset(sta->supported_rates, 0,
			       sizeof(sta->supported_rates));
			memcpy(sta->supported_rates, u, ileft);
			prism2_check_tx_rates(sta);

			u += ileft;
			left -= ileft;
		}

		if (left > 0) {
			PDEBUG2(DEBUG_AP, "\n  extra data (%d bytes) [", left);
			while (left > 0) {
				PDEBUG2(DEBUG_AP, "<%02x>", *u);
				u++; left--;
			}
			PDEBUG2(DEBUG_AP, "]\n");
		}
	} else {
		PDEBUG2(DEBUG_AP, " - frame underflow (left=%d)\n",
			left);
		atomic_dec(&sta->users);
		return;
	}

	/* get a unique AID */
	if (sta->aid > 0)
		PDEBUG2(DEBUG_AP, " - old AID %d\n", sta->aid);
	else {
		spin_lock_irqsave(&local->ap->sta_table_lock, flags);
		for (sta->aid = 1; sta->aid <= MAX_AID_TABLE_SIZE; sta->aid++)
			if (local->ap->sta_aid[sta->aid - 1] == NULL)
				break;
		if (sta->aid > MAX_AID_TABLE_SIZE) {
			sta->aid = 0;
			spin_unlock_irqrestore(&local->ap->sta_table_lock,
					       flags);
			resp = WLAN_STATUS_AP_UNABLE_TO_HANDLE_NEW_STA;
			PDEBUG2(DEBUG_AP, " - no room for more AIDs\n");
		} else {
			local->ap->sta_aid[sta->aid - 1] = sta;
			spin_unlock_irqrestore(&local->ap->sta_table_lock,
					       flags);
			PDEBUG2(DEBUG_AP, " - new AID %d\n", sta->aid);
		}
	}

 fail:
	txdesc = (struct hfa384x_tx_frame *) rxdesc;

	txdesc->frame_control = __cpu_to_le16(
		(WLAN_FC_TYPE_MGMT << 2) |
		((send_deauth ? WLAN_FC_STYPE_DEAUTH :
		  (reassoc ? WLAN_FC_STYPE_REASSOC_RESP :
		   WLAN_FC_STYPE_ASSOC_RESP)) << 4));
	/* Addr1 = DA */
	memcpy(txdesc->addr1, txdesc->addr2, 6);
	/* Addr2 = SA */
	memcpy(txdesc->addr2, dev->dev_addr, 6);
	/* Addr3 = BSSID - already set */

	pos = (u16 *) body;

	if (send_deauth) {
		*pos = __cpu_to_le16(WLAN_REASON_STA_REQ_ASSOC_WITHOUT_AUTH);
		pos++;
	} else {
		/* FIX: CF-Pollable and CF-PollReq should be set to match the
		 * values in beacons/probe responses */
		*pos = __cpu_to_le16(WLAN_CAPABILITY_ESS); /* capability */
		pos++;

		/* status_code */
		*pos = __cpu_to_le16(resp);
		pos++;

		*pos = __cpu_to_le16((sta && sta->aid > 0 ? sta->aid : 0) |
				     BIT(14) | BIT(15)); /* AID */
		pos++;

		/* Supported rates (Information element) */
		p = (char *) pos;
		*p++ = WLAN_EID_SUPP_RATES;
		*p++ = 4; /* len */
		*p++ = 0x82; /* 1 Mbps, base set */
		*p++ = 0x84; /* 2 Mbps, base set */
		*p++ = 0x0b; /* 5.5 Mbps */
		*p++ = 0x16; /* 11 Mbps */
	}

	prism2_send_mgmt(dev, txdesc, body, send_deauth ? 2 : sizeof(body), 1);

	if (sta) {
		if (resp == WLAN_STATUS_SUCCESS) {
			sta->flags |= WLAN_STA_ASSOC;
			sta->last_assoc = jiffies;
		}
		atomic_dec(&sta->users);
	}
}


/* Called only from software IRQ */
static void handle_deauth(local_info_t *local, struct hfa384x_rx_frame *rxdesc)
{
	struct net_device *dev = local->dev;
	char *body = (char *) (rxdesc + 1);
	int len;
	u16 reason_code, *pos;
	struct sta_info *sta = NULL;
	unsigned long flags;

	len = __le16_to_cpu(rxdesc->data_len);

	if (len < 2) {
		printk("handle_deauth - too short payload (len=%d)\n", len);
		return;
	}

	pos = (u16 *) body;
	reason_code = __le16_to_cpu(*pos);

	PDEBUG(DEBUG_AP, "%s: deauthentication: " MACSTR " len=%d, "
	       "reason_code=%d\n", dev->name, MAC2STR(rxdesc->addr2), len,
	       reason_code);

	spin_lock_irqsave(&local->ap->sta_table_lock, flags);
	sta = ap_get_sta(local->ap, rxdesc->addr2);
	if (sta != NULL)
		sta->flags &= ~(WLAN_STA_AUTH | WLAN_STA_ASSOC);
	spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);
	if (sta == NULL) {
		printk("%s: deauthentication from " MACSTR ", "
	       "reason_code=%d, but STA not authenticated\n", dev->name,
		       MAC2STR(rxdesc->addr2), reason_code);
	}
}


/* Called only from software IRQ */
static void handle_disassoc(local_info_t *local,
			    struct hfa384x_rx_frame *rxdesc)
{
	struct net_device *dev = local->dev;
	char *body = (char *) (rxdesc + 1);
	int len;
	u16 reason_code, *pos;
	struct sta_info *sta = NULL;
	unsigned long flags;

	len = __le16_to_cpu(rxdesc->data_len);

	if (len < 2) {
		printk("handle_disassoc - too short payload (len=%d)\n", len);
		return;
	}

	pos = (u16 *) body;
	reason_code = __le16_to_cpu(*pos);

	PDEBUG(DEBUG_AP, "%s: disassociation: " MACSTR " len=%d, "
	       "reason_code=%d\n", dev->name, MAC2STR(rxdesc->addr2), len,
	       reason_code);

	spin_lock_irqsave(&local->ap->sta_table_lock, flags);
	sta = ap_get_sta(local->ap, rxdesc->addr2);
	if (sta != NULL)
		sta->flags &= ~WLAN_STA_ASSOC;
	spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);
	if (sta == NULL) {
		printk("%s: disassociation from " MACSTR ", "
	       "reason_code=%d, but STA not authenticated\n", dev->name,
		       MAC2STR(rxdesc->addr2), reason_code);
	}
}


#ifdef PRISM2_NULLFUNC_ACK
/* Called only from software IRQ */
static void ap_handle_data_nullfunc(local_info_t *local,
				    struct hfa384x_rx_frame *rxdesc)
{
	struct net_device *dev = local->dev;
	struct hfa384x_tx_frame *txdesc;

	/* some STA f/w's seem to require control::ACK frame for
	 * data::nullfunc, but Prism2 f/w does not send this..
	 * send control::ACK for the data::nullfunc */

	txdesc = (struct hfa384x_tx_frame *) rxdesc;

	/* Addr1 = RA */
	memcpy(txdesc->addr1, txdesc->addr2, 6);
	/* control:ACK does not have addr2 or addr3 */
	memset(txdesc->addr2, 0, 6);
	memset(txdesc->addr3, 0, 6);

	txdesc->frame_control =
		__cpu_to_le16((WLAN_FC_TYPE_CTRL << 2) |
			      (WLAN_FC_STYPE_ACK << 4));

	printk(KERN_DEBUG "Sending control::ACK for data::nullfunc\n");
	prism2_send_mgmt(dev, txdesc, NULL, 0, 0);
}
#endif /* PRISM2_NULLFUNC_ACK */


/* Called only from software IRQ */
static void ap_handle_dropped_data(local_info_t *local,
				   struct hfa384x_rx_frame *rxdesc)
{
	struct net_device *dev = local->dev;
	struct sta_info *sta;
	struct hfa384x_tx_frame *txdesc;
	u16 reason;
	unsigned long flags;

	spin_lock_irqsave(&local->ap->sta_table_lock, flags);
	sta = ap_get_sta(local->ap, rxdesc->addr2);
	if (sta)
		atomic_inc(&sta->users);
	spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);
	if (sta != NULL && (sta->flags & WLAN_STA_ASSOC)) {
		PDEBUG(DEBUG_AP, "ap_handle_dropped_data: STA is now okay?\n");
		atomic_dec(&sta->users);
		return;
	}

	txdesc = (struct hfa384x_tx_frame *) rxdesc;

	/* Addr1 = DA */
	memcpy(txdesc->addr1, txdesc->addr2, 6);
	/* Addr2 = SA */
	memcpy(txdesc->addr2, dev->dev_addr, 6);
	/* Addr3 = BSSID */
	memcpy(txdesc->addr3, dev->dev_addr, 6);

	if (sta == NULL || !(sta->flags & WLAN_STA_AUTH)) {
		/* send deauth. frame */
		txdesc->frame_control =
			__cpu_to_le16((WLAN_FC_TYPE_MGMT << 2) |
				      (WLAN_FC_STYPE_DEAUTH << 4));
	} else {
		/* send disassoc. frame */
		txdesc->frame_control =
			__cpu_to_le16((WLAN_FC_TYPE_MGMT << 2) |
				      (WLAN_FC_STYPE_DISASSOC << 4));
	}

	reason = __cpu_to_le16(WLAN_REASON_CLASS3_FRAME_FROM_NONASSOC_STA);

	prism2_send_mgmt(dev, txdesc, (char *) &reason, sizeof(reason), 1);

	if (sta)
		atomic_dec(&sta->users);
}


/* Called only from software IRQ */
static void handle_pspoll(local_info_t *local,
			  struct hfa384x_rx_frame *rxdesc)
{
	struct net_device *dev = local->dev;
	struct sta_info *sta;
	u16 aid;
	struct sta_buffer_frame *fbuf;
	unsigned long flags;

	PDEBUG(DEBUG_PS, "handle_pspoll: BSSID=" MACSTR ", TA=" MACSTR "\n",
	       MAC2STR(rxdesc->addr1), MAC2STR(rxdesc->addr2));

	if (memcmp(rxdesc->addr1, dev->dev_addr, 6)) {
		PDEBUG(DEBUG_AP, "handle_pspoll - addr1(BSSID)=" MACSTR
		       " not own MAC\n", MAC2STR(rxdesc->addr1));
		return;
	}

	aid = __le16_to_cpu(rxdesc->duration_id);
	if ((aid & (BIT(15) | BIT(14))) != (BIT(15) | BIT(14))) {
		PDEBUG(DEBUG_PS, "   PSPOLL and AID[15:14] not set\n");
		return;
	}
	aid &= ~BIT(15) & ~BIT(14);
	if (aid == 0 || aid > MAX_AID_TABLE_SIZE) {
		PDEBUG(DEBUG_PS, "   invalid aid=%d\n", aid);
		return;
	}
	PDEBUG(DEBUG_PS, "   aid=%d\n", aid);

	spin_lock_irqsave(&local->ap->sta_table_lock, flags);
	sta = ap_get_sta(local->ap, rxdesc->addr2);
	if (sta)
		atomic_inc(&sta->users);
	spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);

	if (sta == NULL) {
		PDEBUG(DEBUG_PS, "   STA not found\n");
		return;
	}
	if (sta->aid != aid) {
		PDEBUG(DEBUG_PS, "   received aid=%i does not match with "
		       "assoc.aid=%d\n", aid, sta->aid);
		return;
	}

	/* FIX: todo:
	 * - add timeout for buffering (clear aid in TIM vector if buffer timed
	 *   out (expiry time must be longer than ListenInterval for
	 *   the corresponding STA; "8802-11: 11.2.1.9 AP aging function"
	 * - what to do, if buffered, pspolled, and sent frame is not ACKed by
	 *   sta; store buffer for later use and leave TIM aid bit set? use
	 *   TX event to check whether frame was ACKed?
	 */

	for (;;) {
		int len, res, idx;

		spin_lock_irqsave(&local->ap->sta_table_lock, flags);
		fbuf = sta->tx_buf_head;
		if (fbuf != NULL) {
			sta->tx_buf_head = fbuf->next;
			if (fbuf->next == NULL)
				sta->tx_buf_tail = NULL;
			sta->buffer_count--;
		}
		spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);

		if (fbuf == NULL) {
			if (sta->buffer_count != 0) {
				printk("STA frame buffer empty, but "
				       "buffer_count=%d\n", sta->buffer_count);
			}
			break;
		}

		/* send buffered frame .. */
		PDEBUG(DEBUG_PS, "Sending buffered frame to STA after PS POLL"
		       " (buffer_count=%d)\n", sta->buffer_count);

		len = fbuf->skb->len;

		idx = prism2_get_txfid_idx(local);
		if (idx < 0)
			goto fail;

		if (sta->buffer_count > 0) {
			/* indicate to STA that more frames follow */
			fbuf->txdesc.frame_control |=
				__cpu_to_le16(WLAN_FC_MOREDATA);
		}

		if (local->frame_dump & PRISM2_DUMP_TX_HDR)
			prism2_dump_tx_header(dev->name, &fbuf->txdesc);

		spin_lock_bh(&local->baplock);
		res = hfa384x_setup_bap(dev, BAP0, local->txfid[idx], 0);
		if (!res)
			res = hfa384x_to_bap(dev, BAP0, &fbuf->txdesc,
					     sizeof(fbuf->txdesc));
		if (!res)
			res = hfa384x_to_bap(dev, BAP0, &snap_header,
					     sizeof(snap_header));
		if (!res)
			res = hfa384x_to_bap(dev, BAP0, fbuf->skb->data + 12,
					     fbuf->skb->len - 12);
		spin_unlock_bh(&local->baplock);
		if (!res)
			res = prism2_transmit(dev, idx);

		if (res) {
			local->intransmitfid[idx] = PRISM2_TXFID_EMPTY;
			printk("%s: handle_pspoll - to BAP0 failed\n",
			       dev->name);
			if (res == -ETIMEDOUT)
				prism2_hw_reset(dev);
			goto fail;
		}

		add_tx_bytes(&local->stats, len);

		sta->tx_packets++;
		sta->tx_bytes += len;
		sta->last_rxtx = jiffies;

	fail:
		dev_kfree_skb(fbuf->skb);
		kfree(fbuf);

		if (sta->flags & WLAN_STA_PS) {
			/* send only one buffered packet per PS Poll */
			/* FIX: should ignore further PS Polls until the
			 * buffered packet that was just sent is acknowledged
			 * (Tx or TxExc event) */
			break;
		}
	}


	if (sta->buffer_count == 0) {
		/* try to clear aid from TIM */
		prism2_set_tim(dev, aid, 0);
	}

	atomic_dec(&sta->users);
}


/* Called only from software IRQ */
static void handle_ap_item(local_info_t *local,
			   struct hfa384x_rx_frame *rxdesc)
{
	struct net_device *dev = local->dev;
	u16 fc, type, stype;

	fc = __le16_to_cpu(rxdesc->frame_control);
	type = WLAN_FC_GET_TYPE(fc);
	stype = WLAN_FC_GET_STYPE(fc);

	if (type == WLAN_FC_TYPE_DATA) {
		PDEBUG(DEBUG_AP, "handle_ap_item - data frame\n");

		if (!(fc & WLAN_FC_TODS) || (fc & WLAN_FC_FROMDS)) {
			PDEBUG(DEBUG_AP, "   not ToDS frame (fc=0x%04x)\n",
			       fc);
			goto done;
		}

		if (memcmp(rxdesc->addr1, dev->dev_addr, 6)) {
			PDEBUG(DEBUG_AP, "handle_ap_item - addr1(BSSID)="
			       MACSTR " not own MAC\n",
			       MAC2STR(rxdesc->addr1));
			goto done;
		}

#ifdef PRISM2_NULLFUNC_ACK
		if (stype == WLAN_FC_STYPE_NULLFUNC)
			ap_handle_data_nullfunc(local, rxdesc);
		else
#endif
			ap_handle_dropped_data(local, rxdesc);
		goto done;
	}

	if (type == WLAN_FC_TYPE_CTRL &&
	    stype == WLAN_FC_STYPE_PSPOLL) {
		handle_pspoll(local, rxdesc);
		goto done;
	}

	if (type != WLAN_FC_TYPE_MGMT) {
		PDEBUG(DEBUG_AP, "handle_ap_item - not a management frame?\n");
		goto done;
	}

	if (memcmp(rxdesc->addr1, dev->dev_addr, 6)) {
		PDEBUG(DEBUG_AP, "handle_ap_item - addr1(DA)=" MACSTR
		       " not own MAC\n", MAC2STR(rxdesc->addr1));
		goto done;
	}

	if (memcmp(rxdesc->addr3, dev->dev_addr, 6)) {
		PDEBUG(DEBUG_AP, "handle_ap_item - addr3(BSSID)=" MACSTR
		       " not own MAC\n", MAC2STR(rxdesc->addr3));
		goto done;
	}

	switch (stype) {
	case WLAN_FC_STYPE_ASSOC_REQ:
		handle_assoc(local, rxdesc, 0);
		break;
	case WLAN_FC_STYPE_ASSOC_RESP:
		PDEBUG(DEBUG_AP, "==> ASSOC RESP (ignored)\n");
		break;
	case WLAN_FC_STYPE_REASSOC_REQ:
		handle_assoc(local, rxdesc, 1);
		break;
	case WLAN_FC_STYPE_REASSOC_RESP:
		PDEBUG(DEBUG_AP, "==> REASSOC RESP (ignored)\n");
		break;
	case WLAN_FC_STYPE_ATIM:
		PDEBUG(DEBUG_AP, "==> ATIM (ignored)\n");
		break;
	case WLAN_FC_STYPE_DISASSOC:
		handle_disassoc(local, rxdesc);
		break;
	case WLAN_FC_STYPE_AUTH:
		handle_authen(local, rxdesc);
		break;
	case WLAN_FC_STYPE_DEAUTH:
		handle_deauth(local, rxdesc);
		break;
	default:
		PDEBUG(DEBUG_AP, "Unknown mgmt frame subtype 0x%02x\n", stype);
		break;
	}

 done:
	kfree(rxdesc);
}


/* Called only from software IRQ */
void handle_ap_queue(void *data)
{
	local_info_t *local = (local_info_t *) data;
	struct hfa384x_rx_frame *rxdesc;
	int i;

	if ((local->link->state & (DEV_PRESENT | DEV_CONFIG)) !=
	    (DEV_PRESENT | DEV_CONFIG)) {
		printk("prism2: handle_ap_queue, but dev not OK\n");
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0))
		MOD_DEC_USE_COUNT;
#endif
		return;
	}

	for (;;) {
		int type = 0;

		rxdesc = NULL;
		spin_lock_irq(&local->ap->ap_queue_lock);
		for (i = 0; i < AP_QUEUE_LEN; i++) {
			if (local->ap->ap_queued_items[i] != NULL) {
				rxdesc = (struct hfa384x_rx_frame *)
					local->ap->ap_queued_items[i];
				type = local->ap->ap_queued_type[i];
				local->ap->ap_queued_items[i] = NULL;
			}
		}
		spin_unlock_irq(&local->ap->ap_queue_lock);

		if (rxdesc == NULL)
			break;

		if (type == AP_QUEUED_RXDESC)
			handle_ap_item(local, rxdesc);
		else
			printk("handle_ap_queue: unknown type %d\n", type);
	}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0))
	MOD_DEC_USE_COUNT;
#endif
}


/* Called only from hardware IRQ */
void hostap_rx(struct net_device *dev, struct hfa384x_rx_frame *rxdesc,
	       int force)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 fc, type, stype, len;
	char *buf;
	int res, i;

	fc = __le16_to_cpu(rxdesc->frame_control);
	type = WLAN_FC_GET_TYPE(fc);
	stype = WLAN_FC_GET_STYPE(fc);

	/* skip control, data, and mgmt::beacon frames */
	if (!force &&
	    ((type != WLAN_FC_TYPE_MGMT && type != WLAN_FC_TYPE_CTRL) ||
	     (type == WLAN_FC_TYPE_MGMT && stype == WLAN_FC_STYPE_BEACON))) {
#ifndef final_version
		if (type != WLAN_FC_TYPE_MGMT)
			PDEBUG(DEBUG_AP, "%s: hostap_rx - hostap ignored frame"
			       " (type=0x%02x, subtype=0x%02x)\n", dev->name,
			       type, stype);
#endif
		/* FIX: might collect mgmt::beacon statistics and a list of
		 * other APs that are heard in the neighborhood */
		return;
	}

#if 0
	PDEBUG(DEBUG_AP, "%s: hostap_rx - queuing type=0x%02x stype=0x%02x\n",
	       dev->name, type, stype);
#endif

	len = __le16_to_cpu(rxdesc->data_len);

	buf = (char *) kmalloc(sizeof(*rxdesc) + len, GFP_ATOMIC);
	if (buf == NULL) {
		printk("%s: hostap_rx - could not allocate memory for queued "
		       "item\n", dev->name);
		return;
	}
	memcpy(buf, rxdesc, sizeof(*rxdesc));
	res = hfa384x_from_bap(dev, BAP1, buf + sizeof(*rxdesc), len);
	if (res) {
		printk("%s: data copy from BAP1 failed %d\n",
		       dev->name, res);
		kfree(buf);
		return;
	}

	local->stats.rx_packets++;

	spin_lock(&local->ap->ap_queue_lock);
	for (i = 0; i < AP_QUEUE_LEN && local->ap->ap_queued_items[i] != NULL;
	     i++) {}
	if (i < AP_QUEUE_LEN) {
		local->ap->ap_queued_items[i] = buf;
		local->ap->ap_queued_type[i] = AP_QUEUED_RXDESC;
	}
	spin_unlock(&local->ap->ap_queue_lock);
	if (i >= AP_QUEUE_LEN) {
		printk("%s: AP queue full - dropping new message\n",
		       dev->name);
		kfree(buf);
		return;
	}

/* tq_scheduler was removed in 2.4.0-test12 */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
	queue_task(&local->ap->ap_queue, &tq_scheduler);
#else
	MOD_INC_USE_COUNT;
	if (schedule_task(&local->ap->ap_queue) == 0)
		MOD_DEC_USE_COUNT;
#endif
}


/* Called only from hardware IRQ */
void schedule_packet_send(local_info_t *local, struct sta_info *sta)
{
	int i;
	struct hfa384x_rx_frame *rxdesc;

	if (sta->buffer_count == 0)
		return;

	rxdesc = (struct hfa384x_rx_frame *)
		kmalloc(sizeof(*rxdesc), GFP_ATOMIC);
	if (rxdesc == NULL) {
		printk("schedule_packet_send: kmalloc failed\n");
		return;
	}
	memset(rxdesc, 0, sizeof(*rxdesc));
	rxdesc->frame_control = __cpu_to_le16((WLAN_FC_TYPE_CTRL << 2) |
					      (WLAN_FC_STYPE_PSPOLL << 4));
	memcpy(rxdesc->addr1, local->dev->dev_addr, 6);
	memcpy(rxdesc->addr2, sta->addr, 6);
	rxdesc->duration_id = __cpu_to_le16(sta->aid | BIT(15) | BIT(14));

	spin_lock(&local->ap->ap_queue_lock);
	for (i = 0; i < AP_QUEUE_LEN && local->ap->ap_queued_items[i] != NULL;
	     i++) {}
	if (i < AP_QUEUE_LEN)
		local->ap->ap_queued_items[i] = rxdesc;
	spin_unlock(&local->ap->ap_queue_lock);
	if (i >= AP_QUEUE_LEN) {
		printk("schedule_packet_send: AP queue full\n");
		kfree(rxdesc);
		return;
	}

	PDEBUG(DEBUG_PS, "Scheduling buffered packet delivery for STA\n");

/* tq_scheduler was removed in 2.4.0-test12 */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
	queue_task(&local->ap->ap_queue, &tq_scheduler);
#else
	MOD_INC_USE_COUNT;
	if (schedule_task(&local->ap->ap_queue) == 0)
		MOD_DEC_USE_COUNT;
#endif
}


#ifdef WIRELESS_EXT
int prism2_ap_get_sta_qual(local_info_t *local, struct sockaddr addr[],
			    struct iw_quality qual[])
{
	struct ap_data *ap = local->ap;
	struct list_head *ptr;
	int count = 0;
	unsigned long flags;

	spin_lock_irqsave(&ap->sta_table_lock, flags);

	for (ptr = ap->sta_list.next; ptr != NULL && ptr != &ap->sta_list;
	     ptr = ptr->next) {
		struct sta_info *sta = (struct sta_info *) ptr;

		addr[count].sa_family = AF_UNIX;
		memcpy(addr[count].sa_data, sta->addr, ETH_ALEN);
		qual[count].qual = sta->last_rx_signal - sta->last_rx_silence;
		qual[count].level = sta->last_rx_signal;
		qual[count].noise = sta->last_rx_silence;
		qual[count].updated = sta->last_rx_updated;

		sta->last_rx_updated = 0;
		count++;
		if (count >= IW_MAX_SPY)
			break;
	}
	spin_unlock_irqrestore(&ap->sta_table_lock, flags);

	return count;
}
#endif /* WIRELESS_EXT */
