/*
 * Wireless LAN (IEEE 802.11) link-layer frame sniffer that dumps the received
 * frames into file that can be read to Ethereal.
 * Copyright (c) 2001, SSH Communications Security Corp
 * Jouni Malinen <jkm@ssh.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. See README and COPYING for
 * more details.
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <signal.h>
#include <wtap.h>

#include "ieee80211.h"


static wtap_dumper *wtapdump = NULL;
static int nl_s = -1;


static void handle_sig(int sig)
{
	int wtap_err;

	fprintf(stderr, "handle_sig(%i) - stopping sniffer\n", sig);

	if (nl_s >= 0)
		close(nl_s);

	if (wtapdump != NULL)
		wtap_dump_close(wtapdump, &wtap_err);

	exit(0);
}


int main(int argc, char *argv[])
{
	int wtap_err;
	struct wtap_pkthdr pkthdr;
	struct sockaddr_nl nladdr;
	int offset, i;
	int skip_fcserr = 0;
	int skip_beacon = 0;

	for (i = 1; i < argc; i++) {
		if (strcmp(argv[i], "-h") == 0) {
			fprintf(stderr,
				"prism2ethereal [-e]\n"
				"  -e = skip FCSErr (RX errors), i.e., do not "
				"write them to dump output\n"
				"  -b = skip beacons\n");
			exit(0);
		} else if (strcmp(argv[i], "-e") == 0) {
			fprintf(stderr, "Skipping FCSErr\n");
			skip_fcserr = 1;
		} else if (strcmp(argv[i], "-b") == 0) {
			fprintf(stderr, "Skipping beacons\n");
			skip_beacon = 1;
		}
	}

	wtapdump = wtap_dump_fdopen(STDOUT_FILENO, WTAP_FILE_PCAP,
				    WTAP_ENCAP_IEEE_802_11, 2344, &wtap_err);

	if (wtapdump == NULL) {
		fprintf(stderr, "wtap_dump_fdopen failed; err=%i\n", wtap_err);
		exit(1);
	}

	nl_s = socket(AF_NETLINK, SOCK_RAW, NETLINK_USERSOCK);
	if (nl_s < 0) {
		perror("socket");
		wtap_dump_close(wtapdump, &wtap_err);
		exit(-1);
	}

	memset(&nladdr, 0, sizeof(nladdr));
	nladdr.nl_family = AF_NETLINK;
	nladdr.nl_groups =  PRISM2_MONITOR_GROUP;
	if (bind(nl_s, (struct sockaddr *) &nladdr, sizeof(nladdr)) < 0) {
		perror("bind");
		close(nl_s);
		wtap_dump_close(wtapdump, &wtap_err);
		exit(-1);
	}

	signal(SIGINT, handle_sig);
	signal(SIGTERM, handle_sig);
	signal(SIGHUP, handle_sig);

	offset = (int) &((struct hfa384x_rx_frame *) 0)->frame_control;

	fprintf(stderr, "Dumping frames to stdout - Use Ctrl-C (or SIGTERM, "
		"SIGINT, SIGHUP) to stop\n");

	setpriority(PRIO_PROCESS, 0, -20);

	for (;;) {
		unsigned char buf[8192], *start;
		int len, dlen;
		struct hfa384x_rx_frame *rx;
		unsigned int fc;

		len = recvfrom(nl_s, buf, sizeof(buf), 0, NULL, NULL);
		if (len < 0) {
			perror("recvfrom");
			break;
		}

		rx = (struct hfa384x_rx_frame *) buf;
		fc = le_to_host16(rx->frame_control);

		fprintf(stderr, "RX frame: len=%i", len);

		if (rx->status & BIT(0)) {
			if (skip_fcserr) {
				fprintf(stderr, " (FCSErr - skipping)\n");
				continue;
			}
			fprintf(stderr, " (FCSErr)");
		}
		if (WLAN_FC_GET_TYPE(fc) == WLAN_FC_TYPE_MGMT &&
		    WLAN_FC_GET_STYPE(fc) == WLAN_FC_STYPE_BEACON) {
			if (skip_beacon) {
				fprintf(stderr, " (beacon - skipping)\n");
				continue;
			}
			fprintf(stderr, " (beacon");
		}
		fprintf(stderr, "\n");

		dlen = len - offset;
		start = buf + offset;

#if 1
		{
			dlen -= 22; /* addr4, data_len, 802.3 */
			memmove(start + 22, start, 2+2+6+6+6+2);
			start += 22;
		}
#endif

		gettimeofday(&pkthdr.ts, NULL);
		pkthdr.len = pkthdr.caplen = dlen;
		pkthdr.pkt_encap = WTAP_ENCAP_IEEE_802_11;
		wtap_dump(wtapdump, &pkthdr, NULL, start, &wtap_err);
		if (wtap_err < 0) {
			fprintf(stderr, "wrap_dump error %i\n", wtap_err);
			break;
		}
	}

	close(nl_s);
	wtap_dump_close(wtapdump, &wtap_err);

	return 0;
}
